/*
 * Editor.java
 *
 * Copyright (c) 2000-2001, The University of Sheffield.
 *
 * This file is part of GATE (see http://gate.ac.uk/), and is free
 * software, licenced under the GNU Library General Public License,
 * Version 2, June1991.
 *
 * A copy of this licence is included in the distribution in the file
 * licence.html, and is also available at http://gate.ac.uk/gate/licence.html.
 *
 * Valentin Tablan, October 2000
 *
 * $Id: Editor.java,v 1.9 2001/11/18 15:16:24 valyt Exp $
 */

/*
 * Modified version that loads fonts from file/resource and activates
 * the input method editor. (Eric Auer, 2003.)
 */

package guk;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.undo.*;
import java.beans.*;
import java.io.*;
import java.util.Locale;
import java.util.*;

import java.awt.im.*; /* InputContext etc. */
import java.awt.im.spi.InputMethod; // manual loading
import java.awt.im.spi.InputMethodDescriptor; // manual loading

import uk.ac.gate.guk.im.GateIM; // changed from: guk.im.GateIM;
import uk.ac.gate.guk.im.GateIMDescriptor; // changed from: guk.im.GateIMDescriptor;

import guk.editIM.*; /* *** */

/**
 * A simple text editor included here to demonstrate the capabilities of the GUK
 * package.
 *
 * @author             <a href="http://www.gate.ac.uk/people/">The Gate Team</a>
 * @version            1.0
 */
public class Editor extends JFrame {
  JPanel contentPane;
  JMenuBar jMenuBar1 = new JMenuBar();
  JMenu jMenuFile = new JMenu();
  JMenu jMenuEdit = new JMenu();
  JMenu jMenuHelp = new JMenu();
  JMenu jMenuIM = null;
  JMenuItem jMenuHelpAbout = new JMenuItem();
  JToolBar jToolBar = new JToolBar();
  JTextPane textPane = new JTextPane();
  JMenu jMenuOptions = new JMenu();
  JComboBox fontsComboBox;
  JComboBox sizeComboBox;
  JCheckBoxMenuItem jCheckBoxMenuItemKeyboardMap = new JCheckBoxMenuItem();
  Action openAction, saveAction, saveAsAction, closeAction,
         exitAction, undoAction, redoAction, cutAction, copyAction,
         pasteAction, attributesChangedAction;
  /**
   * The current open file
   */
  File file = null;
  /**
   * The file chooser used in all operations requiring the user to select a file
   */
  JFileChooser filer = new JFileChooser();
  /**
   * handle to store the current directory of the file choser, null is home.
   */
  File filerCWD = null;
  /**
   * The main frame
   */
  JFrame frame;
  UndoManager undoManager = new UndoManager();
  /**
   * has the current document changed since the last save?
   */
  boolean docChanged = false;

  /**
   * Using Erics GUK IM user interface (uses Unicode fonts).
   */
  EditIM editIM = null; /* *** */


  /**
   * Experimental: Have a global InputMethodDescriptor and InputMethod
   * and control object for manual loading.
   */
  InputMethodDescriptor imd = null;
  InputMethod im = null;
  Object imControl = null;

  /**
   * Activate this to enable several kludges to enforce using dynamic fonts,
   * not needed normally. Reason: loading fonts by creating a FontLoader
   * seems to add fonts and families to the GraphicsEnvironment.
   * getLocalGraphicsEnvironment().getAvailableFontFamilyNames()
   * implicitly, a simple createFont() seems to be enough! Further,
   * FontOptimizer normally autodetects when FontLoader is needed.
   */
  boolean useDynFontSpecials = false;

  /**
   * Stores the "best" font combo box index that we have found as default
   */
  int stdFontIndex = 0;

  /**
   * Construct the frame
   */
  public Editor() {

    frame = this;
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE); /* *** */

    enableEvents(AWTEvent.WINDOW_EVENT_MASK);
    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }

    frame.validate();

    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize = getSize();
    if (frameSize.height > screenSize.height) {
      frameSize.height = screenSize.height;
    }
    if (frameSize.width > screenSize.width) {
      frameSize.width = screenSize.width;
    }
    setLocation((screenSize.width - frameSize.width) / 2,
              (screenSize.height - frameSize.height) / 2);
    setVisible(true);

  }// public Editor()


  /**
   * Component initialization
   */
  private void jbInit() throws Exception {
    java.util.List installedLocales = new ArrayList();
    try{
      //if this fails guk is not present
      Class.forName("uk.ac.gate.guk.im.GateIMDescriptor");
        // was: guk.im.GateIMDescriptor
      //add the Gate input methods
      imd = new /* uk.ac.gate.guk.im */ GateIMDescriptor(); /* *** */
      installedLocales.addAll(Arrays.asList(imd.getAvailableLocales()));
      im = imd.createInputMethod(); /* *** */
      imControl = im.getControlObject();
      if ((imControl == null) || !(imControl instanceof GateIM)) {
	imControl = null;
	System.out.println("Could not get GUK IM control object");
      }
      System.out.println("GUK IM loaded, " + installedLocales.size()
	+ " Locales available");
    }catch(Exception e){
      //something happened; most probably guk not present.
      //just drop it, is not vital.
    }
    try{
      //add the MPI IMs
      //if this fails mpi IM is not present
      Class.forName("mpi.alt.java.awt.im.spi.lookup.LookupDescriptor");

// ***      installedLocales.addAll(Arrays.asList(
// ***            new mpi.alt.java.awt.im.spi.lookup.LookupDescriptor().
// ***            getAvailableLocales()));
      System.out.println("MPI IM *** NOT *** loaded"); /* *** */
    }catch(Exception e){
      //something happened; most probably MPI not present.
      //just drop it, is not vital.
    }
    Collections.sort(installedLocales, new Comparator(){
      public int compare(Object o1, Object o2){
        return ((Locale)o1).getDisplayName().compareTo(((Locale)o2).getDisplayName());
      }
    });
    JMenuItem item;
    if(!installedLocales.isEmpty()) {
      jMenuIM = new JMenu("Input methods");
      ButtonGroup bg = new ButtonGroup();
      Iterator localIter = installedLocales.iterator();
      while(localIter.hasNext()){
        Locale aLocale = (Locale)localIter.next();
        item = new LocaleSelectorMenuItem(aLocale, frame, (GateIM)imControl);
        jMenuIM.add(item);
        bg.add(item);
      }
    }// if

    undoManager.setLimit(1000);

    filerCWD = new File("./"); /* I prefer ./ to null aka. ~/ */ /* *** */

    //OPEN ACTION
    openAction = new AbstractAction("Open", new ImageIcon(
            guk.Editor.class.getResource("img/openFile.gif"))){
      public void actionPerformed(ActionEvent e){
        int res = JOptionPane.OK_OPTION;
        if(docChanged){
          res = JOptionPane.showConfirmDialog(
                frame,
                "Close unsaved file " +
                (file== null?"Untitled":file.getName()) + "?",
                "Gate",
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.WARNING_MESSAGE);
        }
        if(res == JOptionPane.OK_OPTION){
          filer.setMultiSelectionEnabled(false);
          filer.setDialogTitle("Select file to open...");
          filer.setSelectedFile(null);
          filer.setCurrentDirectory(filerCWD); /* *** */
          filer.setFileFilter(filer.getAcceptAllFileFilter());
          int res1 = filer.showOpenDialog(frame);
          if(res1 == filer.APPROVE_OPTION){
            filerCWD = filer.getCurrentDirectory(); /* *** */
            //we have the file, what's the encoding?
            Object[] encodings = { "UTF-8", "UTF-16BE", "UTF-16LE", "UTF-16",
                                   "ISO-8859-1", "US-ASCII" };
            Object encoding = JOptionPane.showInputDialog(
                      frame,
                      "Please select the encoding for the chosen file",
                      "Gate",
                      JOptionPane.INFORMATION_MESSAGE,
                      null,
                      encodings,
                      encodings[0]
                      );
            if(encoding == null) return;
            file = filer.getSelectedFile();
            try {
              InputStreamReader reader = new InputStreamReader(
                new BufferedInputStream(new FileInputStream(file)),
                (String)encoding);
              textPane.selectAll();
              /* *** save font? *** */
              /// (not done)
              textPane.replaceSelection("");
              textPane.read(reader, null);
              reader.close();

              /* *** restore font or set to default font *** */
              textPane.selectAll();
              fontsComboBox.setSelectedIndex(stdFontIndex);
              System.out.println("Activating font choice number (0..) "
                + stdFontIndex);
              sizeComboBox.setSelectedIndex(3); /* default, see elsewhere */
              textPane.setCaretPosition(0); // cursor to beginning
                // setCaretPosition warps caret, moveCaretPosition selects.

            } catch(FileNotFoundException fnfe) {
              JOptionPane.showMessageDialog(frame,
                                            "Cannot find the file specified!",
                                            "Gate",
                                            JOptionPane.ERROR_MESSAGE);
              file = null;
              docChanged = false;
              updateTitle();
            } catch(UnsupportedEncodingException usee) {
              JOptionPane.showMessageDialog(frame,
                                            "Unsupported encoding!\n" +
                                            "Please choose another.",
                                            "Gate",
                                            JOptionPane.ERROR_MESSAGE);
              file = null;
              docChanged = false;
              updateTitle();
            } catch(IOException ioe) {
              JOptionPane.showMessageDialog(
                                  frame,
                                  "Input/Output error! (wrong encoding?)\n" +
                                  "Please try again.",
                                  "Gate",
                                  JOptionPane.ERROR_MESSAGE);
              file = null;
              docChanged = false;
              updateTitle();
            }
            docChanged = false;
            updateTitle();
          }// selected any file (approved)
        }// accepted closing previous file (if any)
      }// actionPerformed(ActionEvent e)
    };
    openAction.putValue(Action.SHORT_DESCRIPTION, "Open file...");


    //SAVE ACTION
    saveAction = new AbstractAction("Save", new ImageIcon(
            guk.Editor.class.getResource("img/saveFile.gif"))) {
      public void actionPerformed(ActionEvent e){
        if(!docChanged) return; // save not needed
        if(file == null) {
          saveAsAction.actionPerformed(null);
          return;
        }
        //get the encoding
        Object[] encodings = { "UTF-8", "UTF-16BE", "UTF-16LE", "UTF-16",
                                "ISO-8859-1", "US-ASCII" };
        Object encoding = JOptionPane.showInputDialog(
                  frame,
                  "Please select the encoding for the chosen file",
                  "Gate",
                  JOptionPane.INFORMATION_MESSAGE,
                  null,
                  encodings,
                  encodings[0]
                  );
        if(encoding == null) return;
        try {
          OutputStreamWriter writer = new OutputStreamWriter(
              new FileOutputStream(file), (String)encoding);
          writer.write(textPane.getText());
          writer.flush();
          writer.close();
          docChanged = false;
          updateTitle();
        } catch(UnsupportedEncodingException usee) {
          JOptionPane.showMessageDialog(frame,
                                        "Unsupported encoding!\n" +
                                        "Please choose another.",
                                         "Gate",
                                        JOptionPane.ERROR_MESSAGE);
          docChanged = true;
          updateTitle();
        } catch(IOException ioe) {
          JOptionPane.showMessageDialog(frame,
                                        "Input/Output error!\n" +
                                        "Please try again.",
                                        "Gate",
                                        JOptionPane.ERROR_MESSAGE);
          docChanged = true;
          updateTitle();
        }
      }// actionPerformed(ActionEvent e)
    };
    saveAction.putValue(Action.SHORT_DESCRIPTION, "Save...");

    //SAVE AS ACTION
    saveAsAction = new AbstractAction("Save as...", new ImageIcon(
            guk.Editor.class.getResource("img/saveFile.gif"))){
      public void actionPerformed(ActionEvent e) {
        filer.setMultiSelectionEnabled(false);
        filer.setDialogTitle("Select file to save to...");
        filer.setSelectedFile(null);
        filer.setCurrentDirectory(filerCWD); /* *** */
        filer.setFileFilter(filer.getAcceptAllFileFilter());
        int res = filer.showSaveDialog(frame);
        if (res != filer.APPROVE_OPTION) return;
        File newFile = filer.getSelectedFile();
        if(newFile == null) return; // do not even remember CWD
        filerCWD = filer.getCurrentDirectory(); /* *** */
        int res1 = JOptionPane.OK_OPTION;
        if(newFile.exists()){
          res1 = JOptionPane.showConfirmDialog(
                  frame,
                  "Overwrite existing file " + newFile.getName() + "?",
                  "Gate",
                  JOptionPane.OK_CANCEL_OPTION,
                  JOptionPane.WARNING_MESSAGE);
        }
        if(res1 == JOptionPane.OK_OPTION){
          file = newFile; // change filename
          docChanged = true;
          saveAction.actionPerformed(null); // proceed with SAVE
        }
      }// actionPerformed(ActionEvent e)
    };
    saveAsAction.putValue(Action.SHORT_DESCRIPTION, "Save as...");

    //CLOSE ACTION
    closeAction = new AbstractAction("Close", new ImageIcon(
            guk.Editor.class.getResource("img/closeFile.gif"))){
      public void actionPerformed(ActionEvent e){
        int res = JOptionPane.OK_OPTION;
        if(docChanged){
          res = JOptionPane.showConfirmDialog(
                frame,
                "Close unsaved file " +
                (file== null?"Untitled":file.getName()) + "?",
                "Gate",
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.WARNING_MESSAGE);
        }
        if(res == JOptionPane.OK_OPTION){
          textPane.selectAll();
          textPane.replaceSelection("");
          docChanged = false;
          file = null;
          updateTitle();
        }
      }// actionPerformed(ActionEvent e)
    };
    closeAction.putValue(Action.SHORT_DESCRIPTION, "Close...");


    //EXIT ACTION
    exitAction = new AbstractAction("Exit", new ImageIcon(
            guk.Editor.class.getResource("img/exit.gif"))){
      public void actionPerformed(ActionEvent e){
        int res = JOptionPane.OK_OPTION;
        if(docChanged){
          res = JOptionPane.showConfirmDialog(
                frame,
                "Close unsaved file " +
                (file== null?"Untitled":file.getName()) + "?",
                "Gate",
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.WARNING_MESSAGE);
        }
        if(res == JOptionPane.OK_OPTION){
          frame.setVisible(false);
          frame.dispose();
          /* *** additional Object removal by Eric: *** */
          filer = null;
          filerCWD = null;
          file = null;
          undoManager = null;
          editIM = null;
          System.exit(0);
        }
      }// actionPerformed(ActionEvent e)
    };
    exitAction.putValue(Action.SHORT_DESCRIPTION, "Exit...");

    //UNDO ACTION
    undoAction = new AbstractAction("Undo", new ImageIcon(
            guk.Editor.class.getResource("img/undo.gif"))){
      public void actionPerformed(ActionEvent e){
        if(undoManager.canUndo()) undoManager.undo();
      }
    };
     undoAction.setEnabled(undoManager.canUndo());
     undoAction.putValue(Action.SHORT_DESCRIPTION, "Undo...");

    //REDO ACTION
    redoAction = new AbstractAction("Redo", new ImageIcon(
            guk.Editor.class.getResource("img/redo.gif"))){
      public void actionPerformed(ActionEvent e){
        if(undoManager.canRedo()) undoManager.redo();
      }
    };
    redoAction.setEnabled(undoManager.canRedo());
    redoAction.putValue(Action.SHORT_DESCRIPTION, "Redo...");

    //COPY ACTION
    copyAction = new AbstractAction("Copy", new ImageIcon(
            guk.Editor.class.getResource("img/copy.gif"))){
      public void actionPerformed(ActionEvent e){
        textPane.copy();
      }
    };
    copyAction.putValue(Action.SHORT_DESCRIPTION, "Copy...");

    //CUT ACTION
    cutAction = new AbstractAction("Cut", new ImageIcon(
            guk.Editor.class.getResource("img/cut.gif"))){
      public void actionPerformed(ActionEvent e){
        textPane.cut();
      }
    };
    cutAction.putValue(Action.SHORT_DESCRIPTION, "Cut...");

    //PASTE ACTION
    pasteAction = new AbstractAction("Paste", new ImageIcon(
            guk.Editor.class.getResource("img/paste.gif"))){
      public void actionPerformed(ActionEvent e){
        textPane.paste();
      }
    };
    pasteAction.putValue(Action.SHORT_DESCRIPTION, "Paste...");

    //attributesChangedAction (happens very often, even on cursor move)
    // (because our pane can mix Fonts and suspects mixing everywhere.)
    attributesChangedAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        int start = textPane.getSelectionStart();
        int end = textPane.getSelectionEnd();
        int selSize = Integer.parseInt((String)sizeComboBox.getSelectedItem());
        String selFont = (String)fontsComboBox.getSelectedItem();
        // could have been any object, we (String)ed it anyway...
        //change the selection

        MutableAttributeSet as = textPane.getInputAttributes();
        // or use getCharacterAttributes for IMMUTABLE attr. at caret location.
        //
        if (useDynFontSpecials && (FontOptimizer.fontLoader != null)
          && FontOptimizer.fontLoader.getInternalFonts().contains(selFont)) {
          textPane.setFont(
            FontOptimizer.fontLoader.fontSet(selFont, selSize,
              textPane.getFont()));
          // kind of simplistic, MutableAttributeSets have no setFont()
          // so we only can set a dynamic font for the whole pane :-(
          StyleConstants.setFontFamily(as,
            textPane.getFont().getFamily());
              StyleConstants.setFontSize(as, selSize);
        } else {
          StyleConstants.setFontFamily(as, selFont);
          StyleConstants.setFontSize(as, selSize);
        }
        textPane.setCharacterAttributes(as, false /* replace */);
        //
        // apply to selection or to text-which-will-be-typed

        //restore selection
        textPane.setCaretPosition(start);
        textPane.moveCaretPosition(end);
      }// actionPerformed(ActionEvent e)
    };

    textPane.addPropertyChangeListener("document", new PropertyChangeListener(){
      public void propertyChange(PropertyChangeEvent evt){
        undoAction.setEnabled(undoManager.canUndo());
        redoAction.setEnabled(undoManager.canRedo());
        //add the document listener
        textPane.getDocument().addDocumentListener(new DocumentListener(){
          public void insertUpdate(DocumentEvent e){
            changeOccured();
          }
          public void removeUpdate(DocumentEvent e){
            changeOccured();
          }
          public void changedUpdate(DocumentEvent e){
            changeOccured();
          }
          protected void changeOccured(){
            undoAction.setEnabled(undoManager.canUndo());
            undoAction.putValue(Action.SHORT_DESCRIPTION,
                                undoManager.getUndoPresentationName());
            redoAction.setEnabled(undoManager.canRedo());
            redoAction.putValue(Action.SHORT_DESCRIPTION,
                                undoManager.getRedoPresentationName());
            if(docChanged) return;
            else{
              docChanged = true;
              updateTitle();
            }
          }// changeOccured()
        });
        //add the document UNDO listener
        undoManager.discardAllEdits();
        textPane.getDocument().addUndoableEditListener(undoManager);
      }// propertyChange(PropertyChangeEvent evt)
    });

    fontsComboBox = new JComboBox(
                        GraphicsEnvironment.getLocalGraphicsEnvironment().
                        getAvailableFontFamilyNames()
                        ); /* arg is Vector or as here Object[] */


    if ((FontOptimizer.fontLoader != null) && /* *** */
      (!FontOptimizer.fontLoader.getInternalFonts().isEmpty())) {
      Iterator iFontIter = FontOptimizer.fontLoader.
        getInternalFonts().iterator();
      if (useDynFontSpecials)
        while (iFontIter.hasNext())
          fontsComboBox.insertItemAt(iFontIter.next(), 0); /* *** */
      // inserted item could have been any Object. Our handler (String)s
      // it. It becomes new first item. Default setting is <current font>
      // however (set elsewhere)... (insertItem appends at the end)
      // ***** even without useDynFontSpecials, having an active FontLoader
      // ***** add the dynamic fonts to the Available Font Families !? :-)
    } else {
      System.out.println("No dynamically loaded fonts reported");
    }


    fontsComboBox.setEditable(false); /* only given entries */
    fontsComboBox.addActionListener(new ActionListener(){
      public void actionPerformed(ActionEvent e){
        attributesChangedAction.actionPerformed(null);
      }// actionPerformed(ActionEvent e)
    });


    sizeComboBox = new JComboBox(new Object[]{"6", "8", "10", "12", "14", "16",
                                              "18", "20", "22", "24", "26"});
    sizeComboBox.setEditable(true); /* user may enter a value */
    sizeComboBox.addActionListener(new ActionListener(){
      public void actionPerformed(ActionEvent e){
        try {
          Integer.parseInt((String)sizeComboBox.getSelectedItem());
          //fire the action
          // *** should move focus back from combo edit to text pane here!?
          attributesChangedAction.actionPerformed(null);
        } catch(NumberFormatException nfe){
          // invalid input, go to default
          // *** keep this in sync with other set to default cases!
          sizeComboBox.setSelectedIndex(3);
        }
      }//actionPerformed(ActionEvent e)
    });

    //initialisation for the fonts and size combos
    fontsComboBox.setSelectedItem(StyleConstants.getFontFamily(
                                  textPane.getInputAttributes()));

    /* *** store default value! *** */
    stdFontIndex = fontsComboBox.getSelectedIndex();

    sizeComboBox.setSelectedItem(String.valueOf(StyleConstants.getFontSize(
                                  textPane.getInputAttributes())));

    //keep them updated
    textPane.addCaretListener(new CaretListener(){
      public void caretUpdate(CaretEvent e) {
        if(e.getDot() == e.getMark()){
          // *** this is where cursor moves update the combo boxes
          // *** which updates the font / style settings by throwing
          // *** other events, see elsewhere in this class!
          String hereFontFamily = StyleConstants.getFontFamily(
                                        textPane.getCharacterAttributes());
          /* ********** */
          /* slip over to dynamic fonts whenever the family allows it! */
        if (useDynFontSpecials && (FontOptimizer.fontLoader != null) &&
          FontOptimizer.fontLoader.getInternalFonts().contains(
            "Dynamic: " + hereFontFamily) ) /* KLUDGE WARNING */
          hereFontFamily = "Dynamic: " + hereFontFamily;
          /* ********** */
          fontsComboBox.setSelectedItem(hereFontFamily);
          sizeComboBox.setSelectedItem(String.valueOf(StyleConstants.getFontSize(
                                        textPane.getCharacterAttributes())));
        }
      }//caretUpdate(CaretEvent e)
    });

    fontsComboBox.setMaximumSize(new Dimension(150,25));
    //fontsComboBox.setMinimumSize(new Dimension(150,25));
    fontsComboBox.setPreferredSize(new Dimension(150,25));
    //fontsComboBox.setSize(new Dimension(150,25));

    sizeComboBox.setMaximumSize(new Dimension(50,25));
    //sizeComboBox.setMinimumSize(new Dimension(30,25));
    sizeComboBox.setPreferredSize(new Dimension(50,25));
    //sizeComboBox.setSize(new Dimension(30,25));
    sizeComboBox.enableInputMethods(false);

    //setIconImage(Toolkit.getDefaultToolkit().
    //  createImage(EditorFrame.class.getResource("[Your Icon]")));

    contentPane = (JPanel) this.getContentPane();
    contentPane.setLayout(new BorderLayout());
    this.setSize(new Dimension(800, 600));
    updateTitle();

    jMenuFile.setText("File");
    jMenuEdit.setText("Edit");
    jMenuHelp.setText("Help");
    jMenuHelpAbout.setText("About");
    jMenuHelpAbout.addActionListener(new ActionListener()  {
      public void actionPerformed(ActionEvent e) {
        jMenuHelpAbout_actionPerformed(e);
      }
    });
    jMenuOptions.setText("Options");
    jCheckBoxMenuItemKeyboardMap.setText("Keyboard Map");
    jCheckBoxMenuItemKeyboardMap.setSelected(false);
    jCheckBoxMenuItemKeyboardMap.setMnemonic('0');
    jCheckBoxMenuItemKeyboardMap.addActionListener(new ActionListener()  {
      public void actionPerformed(ActionEvent e) {
        jCheckBoxMenuItemKeyboardMap_stateChanged(e);
      }
    });
    jToolBar.add(openAction);
    jToolBar.add(saveAction);
    jToolBar.add(closeAction);
    jToolBar.addSeparator();
    jToolBar.add(undoAction);
    jToolBar.add(redoAction);
    jToolBar.addSeparator();
    jToolBar.add(cutAction);
    jToolBar.add(copyAction);
    jToolBar.add(pasteAction);
    jToolBar.addSeparator();
    jToolBar.add(fontsComboBox);
    jToolBar.addSeparator();
    jToolBar.add(sizeComboBox);

    jToolBar.add(Box.createHorizontalGlue());

    jMenuFile.add(openAction);
    jMenuFile.add(saveAction);
    jMenuFile.add(saveAsAction);
    jMenuFile.add(closeAction);
    jMenuFile.addSeparator();
    jMenuFile.add(exitAction);

    jMenuEdit.add(cutAction);
    jMenuEdit.add(copyAction);
    jMenuEdit.add(pasteAction);
    jMenuEdit.addSeparator();
    jMenuEdit.add(undoAction);
    jMenuEdit.add(redoAction);

    jMenuOptions.add(jCheckBoxMenuItemKeyboardMap);
    if(jMenuIM != null) jMenuOptions.add(jMenuIM);

    jMenuHelp.add(jMenuHelpAbout);

    jMenuBar1.add(jMenuFile);
    jMenuBar1.add(jMenuEdit);
    jMenuBar1.add(jMenuOptions);
    jMenuBar1.add(jMenuHelp);

//    textPane.setEditorKit(new UnicodeStyledEditorKit(GUK.getFontSet()));

    /* *** */
    textPane.setEditorKit(new StyledEditorKit());
    textPane.setFont(new Font(null, Font.PLAIN, 12)); // default / parent font

    // load an optimal Unicode Font:
    textPane.setFont( FontOptimizer.unicodeFont() );

    /** create Input Method Editor window, using our chosen UNICODE
     *  font and allowing it to send events to our GUK IM and FRAME
     *  (hand over NULL font to use default: Only main() of the
     *  Input Method Editor uses FontOptimizer). */
    editIM = new EditIM(textPane.getFont(), imd, im, frame);

    /* *** Update virtual keyboard font as well! *** */
    if ((imControl != null) && (imControl instanceof GateIM)) {
      ((GateIM)imControl).setKeyboardFont(textPane.getFont());
    }

    // UPDATE the fonts and size combos to reflect optimized default!
    fontsComboBox.setSelectedItem(StyleConstants.getFontFamily(
                                  textPane.getInputAttributes()));

    /* *** store NEW default value! *** */
    stdFontIndex = fontsComboBox.getSelectedIndex();

    sizeComboBox.setSelectedItem(String.valueOf(StyleConstants.getFontSize(
                                  textPane.getInputAttributes())));


    this.setJMenuBar(jMenuBar1);
    contentPane.add(jToolBar, BorderLayout.NORTH);
    contentPane.add(new JScrollPane(textPane), BorderLayout.CENTER);
  }// jbInit()



  protected void updateTitle(){
    String title = "Gate Unicode Editor - ";
    if(file != null) title += file.getName();
    else title += "Untitled";
    if(docChanged) title += "*";
    frame.setTitle(title);
  }// updateTitle()


  /**
   * Main method
   */
  public static void main(String[] args) {

    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
    }
    catch(Exception e) {
      e.printStackTrace();
    }

    new Editor();
  }// main


  /**
   * Help | About action performed
   */
  public void jMenuHelpAbout_actionPerformed(ActionEvent e) {
    Editor_AboutBox dlg = new Editor_AboutBox(this);
    Dimension dlgSize = dlg.getPreferredSize();
    Dimension frmSize = getSize();
    Point loc = getLocation();
    dlg.setLocation((frmSize.width - dlgSize.width) / 2 + loc.x,
                    (frmSize.height - dlgSize.height) / 2 + loc.y);
    dlg.setModal(true);
    dlg.show();
  }// jMenuHelpAbout_actionPerformed(ActionEvent e)



  /**
   * Overridden so we can exit when window is closed
   */
  protected void processWindowEvent(WindowEvent e) {
    if (e.getID() == WindowEvent.WINDOW_CLOSING) {
      exitAction.actionPerformed(null);
      System.out.println("Closing!?");
    } else {
      super.processWindowEvent(e);
    }
  }// processWindowEvent(WindowEvent e)



  void jCheckBoxMenuItemKeyboardMap_stateChanged(ActionEvent e) {
    Object imObject = getInputContext().getInputMethodControlObject();
    if(imObject != null && imObject instanceof GateIM){
      ((GateIM)imObject).setMapVisible(
        jCheckBoxMenuItemKeyboardMap.getState());
    } else {
      System.out.println("Could not control global virtual keyboard" +
        "\nIM control object = " + imObject);
      if (imControl != null) {
        ((GateIM)imControl).setMapVisible(
          jCheckBoxMenuItemKeyboardMap.getState());
        System.out.println("updated local virtual keyboard state");
      } else {
        jCheckBoxMenuItemKeyboardMap.setState(false);
      }
    }
  }// void jCheckBoxMenuItemKeyboardMap_stateChanged(ActionEvent e)



/* *** Overridden for this Component: PRIVATE input context *** */
  public InputContext getInputContext() { // null for default
    /* this is called basically for every event in the JFrame */
    if ((im == null) || (imd == null) || (imControl == null)) {
      System.out.println("No GUK available");
      return null;
    }
    // im.activate() ... im.setInputContext(...) ...
    // boolean setLocale(...) ... InputContext.getInstance() ...
    // *** what now??? ***
    // an InputMethodContext: ((GateIM)imControl).getContext();
    return super.getInputContext();
  }


}// class Editor extends JFrame


/////////////////////////////////////////////////////////////////////


class LocaleSelectorMenuItem extends JRadioButtonMenuItem {

  Locale myLocale;

  JRadioButtonMenuItem me;

  Frame frame;

  GateIM imControl;

  public LocaleSelectorMenuItem(Locale locale, Frame pframe, GateIM imC){
    super(locale.getDisplayName()); // ???
    this.frame = pframe;
    me = this;
    myLocale = locale;
    imControl = imC;

    this.addActionListener(new ActionListener()  {
      public void actionPerformed(ActionEvent e) {
        InputContext ic = frame.getInputContext();
        // SIMPLER: me.setSelected(frame.getInputContext().
        //             selectInputMethod(myLocale));
        System.out.println("Locale menu item: old IC L.=" + ic.getLocale()
          + "\nold L.=" + frame.getLocale() + "\nTarget L.=" + myLocale);
        if (ic.selectInputMethod(myLocale)) {
          System.out.println("IC Locale set to: " + ic.getLocale() +
            "\nLocale is: " + frame.getLocale());
          me.setSelected(true);
          return;
        }
        me.setSelected(false);
        if (imControl != null)
          me.setSelected(imControl.setLocale(myLocale));
        System.out.println("setting local IM locale success = " +
          me.isSelected()); // trying frame Locale
        // frame.setLocale(myLocale);
        // System.out.println("Locale set to: " + frame.getLocale());
        return;
      }
    });

  }// LocaleSelectorMenuItem(Locale locale, Frame pframe)

}// class LocaleSelectorMenuItem extends JRadioButtonMenuItem

