/**
 * Helper for the EditIM Input Method Editor: Provides an
 * utility window to use the system clipboard more easily.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*; // Clipboard ...
import javax.swing.*;
import javax.swing.event.*;
// import java.beans.*;
// import java.io.*;


/**
 * Helper for the EditIM Input Method Editor: Provides an
 * utility window to use the system clipboard more easily.
 */
public class ClipWindow {


 /**
  * The MenuHelpers class provides functions like creating menu items
  * and buttons. Initially, no action listener is set.
  */
 MenuHelpers menu = new MenuHelpers(null);


 /**
  * A very kludgy textfield that is supposed to be connected to
  * the system clipboard. Using it normally crashes Java 1.3.1 Unix.
  * May be also useful because you need no clipboard hotkeys to
  * use it.
  */
 JTextArea clipField = new JTextArea("");
 /**
  * A frame for the clipboard user interface that is built around
  * clipField.
  */
 JFrame clipDialog = new JFrame("System clipboard");


 /**
  * The constructor needs a MenuHelpers instance for generic
  * settings like font, action listener and centering base.
  * Instantiates the other helper classes.
  * @param menuHelper Provides functions like creating menu items.
  * Should have the listener and font set.
  */
 public ClipWindow(MenuHelpers menuHelper) {
   menu = menuHelper;
 } // constructor


 /* *** *** */


 /**
  *  <p>
  *  Popup a menu that allows to communicate with the system
  *  clipboard. Actually, the clipboard support would be much more
  *  flexible in Java 1.4 !? (get/set TransferHandler in JComponent).
  *  </p>
  *  <p><i>
  *  Warning: Using the clipboard that way often crashes
  *  Java 1.3.1 Unix. Good question why this happens...
  *  </i></p>
  */
 void clipPopup() {
   if (clipDialog != null)
     clipDialog.dispose(); // remove old instance!
   clipDialog = new JFrame("System clipboard"); // re-init
   JPanel clipDialogPane = (JPanel)clipDialog.getContentPane();
   JPanel clipButtons = new JPanel(new GridLayout(3,1));
   clipDialogPane.setLayout(new BorderLayout());
   menu.syncFont(clipDialogPane);
   //
   clipField.setToolTipText(
     "this is a buffer to exchange text with the system-wide clipboard");
   menu.syncFont(clipField);
   clipDialogPane.add(new JScrollPane(clipField),
     BorderLayout.CENTER);
     // ... an area allowing to see, type and select stuff
     // ... and of course also to copy and paste stuff.
   // clipField.setActionCommand("clipfieldchange ignored");
     // dummy. only for JTextFields, not for JTextAreas.
   clipField.setText("A buffer for\nclipboard contents");
   //
   clipButtons.add(menu.myJButton(
     "copy selection to system clipboard", KeyEvent.VK_C,
     "copy selected text above to system clipboard",
     "CLIPsend"));
   clipButtons.add(menu.myJButton(
     "paste system clipboard value", KeyEvent.VK_V,
     "paste system clipboard at cursor position above",
     "CLIPrecv"));
   clipButtons.add(menu.myJButton("close this window",
     KeyEvent.VK_W, "close this clipboard window",
     "CLIPclose"));
   clipDialogPane.add(clipButtons, BorderLayout.SOUTH);
   //
   clipDialog.pack(); // optimize size
   menu.centerMe(clipDialog);
   clipDialog.show();
   //
 } // clipPopup


 /**
  * Postprocess clipboard dialog results and create commands in
  * proper syntax for our listener.
  * <pre>
  * Accepted commands:
  *   open  - open the window (closing possible other instances)
  *   close - close the window
  *   send  - send selection in this window to clipboard
  *   recv  - add clipboard text to this window
  *   send2 and recv2 are alternative implementations
  * @param type A clipboard related command: send, recv, send2
  * or recv2 or close. Close closes the popup menu.
  */
 public void clipCommand(String type) {
   String command = null;
   if (type.equals("open")) {
     clipPopup(); // create the window with the user interface
   } else if (type.equals("send2")) {
     java.awt.Toolkit.getDefaultToolkit().getSystemClipboard().
       setContents( new StringSelection( clipField.getText() ),
         null /* no owner */ ); // send only selected part
       // an owner would be notified if the clipboard is overwritten
       // see Clipboard class: we can also getContents() from it,
       // receiving a Transferable... then, do try/catch Throwable: x =
       // (String) transferable.getTransferData(DataFlavor.stringFlavor)
   } else if (type.equals("recv2")) {
     try {
       java.awt.datatransfer.Transferable theContent =
         java.awt.Toolkit.getDefaultToolkit().getSystemClipboard().
           getContents(this /* requestor */);
       if ((theContent != null) &&
           (theContent.isDataFlavorSupported(DataFlavor.stringFlavor)))
         clipField.setText( (String)
           theContent.getTransferData( DataFlavor.stringFlavor ) );
     } catch (Throwable thr) {
       java.awt.Toolkit.getDefaultToolkit().beep();
       DebugEditIM.println(1, "no text to receive!");
     }
   } else if (type.equals("send")) {
     clipField.copy();  // SUPPOSED to use the system clipboard
     // only sends the selected part
   } else if (type.equals("recv")) {
     clipField.paste(); // SUPPOSED to use the system clipboard
   } else if (type.equals("close")) {
     clipDialog.dispose(); // free resources again
     // do nothing else, just close
   } else {
     DebugEditIM.println(0, "confused by: " + type);
     return; // do not dispose either
   }
 } // clipCommand


} // class ClipWindow

