/**
 *  Helper class for debug messages. Used by all EditIM parts that
 *  show more than plain "Initializing... done." messages.
 *  2003 by Eric Auer. Has public static DebugEditIM(level, message),
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */

package guk.editIM;

import guk.editIM.*; /* *** */
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.StringTokenizer;

/**
 *  Helper class for debug messages. Used by all EditIM parts that
 *  show more than plain "Initializing... done." messages.
 */
public class DebugEditIM {

  /**
   * debug level: -1 mute, 0 normal, 1 verbose, 2 very verbose, 3 wow!
   * recommended setting: 0 or 1.
   */
  final static int debug = 2;

  /**
   * The only interface is this static method
   * ... which makes creative use of the IMPLEMENTATION DEPENDANT
   * printStackTrace method to print the name of the one who calls it!
   * (from Java 1.4.0 on, we could use methods MEANT for doing this).
   * @param level The verbosity level: If it is at most as high as
   * the debug level, the text will be printed. Too verbose requests
   * are ignored.
   * @param text The text that will be printed, after prefixing the
   * name of the class and method of the caller (but not the package
   * name of it, just the last part of the name).
   */
  public static void println(int level, String text) {
    if (level > debug) return;
      // ignore if it belongs to higher verbosity than current

    String callerS = ""; // name of the caller
    try {
      throw new Throwable(); // how else do we get a stack trace?
    } catch (Throwable t) {
      ByteArrayOutputStream oStream = new ByteArrayOutputStream();
      t.printStackTrace( new PrintStream( oStream ) );
      String stackS = oStream.toString(); // stack trace for this
      StringTokenizer strtok = new StringTokenizer(stackS);
      String onetoken;
      while (strtok.hasMoreTokens()) {
         onetoken = strtok.nextToken("\n"); // read line after line
         if (onetoken.indexOf("DebugEditIM.println") != -1) {
            if (strtok.hasMoreTokens()) callerS = strtok.nextToken();
            // next line after reference to us is reference to caller
            // could leave while now if we are mentioned only once
         } // found something
      } // while
    }
    if (callerS.indexOf("guk.editIM.") >= 0) {
      callerS = callerS.substring(
        callerS.indexOf("guk.editIM.") + "guk.editIM.".length() );
      // strip up to and including "guk.editIM." from method name
    }
    if (callerS.indexOf('(') > 0) {
      callerS = callerS.substring(0, callerS.indexOf('('));
      // strip end part (file name and line number)
    }
    // now we know the caller name, e.g. GlyphTable.processCommand

    System.out.println("EditIM: " + callerS + ": " + text);
    // print to stdout in format: "EditIM: part.method: text"
  }

} // public class DebugEditIM
