/*
 * EditIM.java in guk.editIM: An Editor for Input Methods (for
 * their keymap files, that is).
 * Maybe an idea: Add add/remove assignments to running GUK IM FSM.
 * GUK IM should allow choice lists as MPI IM do.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */

package guk.editIM;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.beans.*;
import java.io.*;
import java.util.Locale;
import java.util.*;

import java.awt.im.*; /* InputContext etc. */
import java.awt.im.spi.InputMethod; // manual loading
import java.awt.im.spi.InputMethodDescriptor; // manual loading

import uk.ac.gate.guk.im.GateIM; // changed from: guk.im.GateIM;
import uk.ac.gate.guk.im.GateIMDescriptor; // changed from: guk.im.GateIMDescriptor;

import guk.editIM.*; /* *** */

/**
 *
 * EditIM.java in guk.editIM: An Editor for Input Methods (for
 * their keymap files, that is). Main file of the
 * <b>Input Method Editor</b>.
 * @author Eric Auer
 */
public class EditIM extends JFrame
  implements ActionListener, ListSelectionListener {

  /**
   * Here, the user can enter a glyph or a glyph number,
   * preferrably in \\u12ab syntax, to scroll to that glyph.
   * Accessed from both look and feel components.
   */
  JTextField jumpField = new JTextField();

  /**
   * The button to select NONE as the current GUK IM.
   * Accessed from both look and feel components, because
   * the feel wants to know whether a blockimusage command
   * came from the button or the checkbox.
   */
  JToggleButton blockIMButton = null; // "block IM" button

  /**
   * Ten buttons, each selecting ONE of the GUK IMs as the
   * IM that is used to edit the tables and the jumpField.
   * Used a lot by look, feel and even the sub menu processor.
   */
  LocaleHotkey[] hotkeys = null; // top 10 hotkeys

  /**
   * a button that serves as a glyph preview window. clicking on it
   * could cause some special actions like "copy glyph to cut buffer"
   * or "add glyph to currently selected table2 row, left buffer" or
   * "send key_typed event for that glyph to our creator"...
   * BONUS: a panel where the sub menu processor can present some
   * interaction components for glyph button related things.
   * glyphChar is the char currently visible on glyphButton, to be
   * used by other related buttons.
   * Used by both the look and the feel (the feel dynamically updates
   * the glyph button).
   */
  JButton glyphButton = new JButton();

  /**
   * The glyphBar is a whole panel where another class can create
   * additional user interface elements. Currently, a palette of
   * glyphs (for fast and easy access, filled with glyphs by the
   * user) exists there.
   * Used by both the look and the feel, in particular the sub menu
   * processor.
   */
  JPanel glyphBar = null;

  /**
   * The char that is currently visible on the glyph preview button.
   * Updated by the the glyph button update (triggered by the list
   * selection listener and read by the action listener.
   */
  char glyphChar = '?';

  /**
   * The first table is implemented in GlyphTable.java as data model.
   * The tables are the main editor component.
   * @see GlyphTable
   */
  JTable table1 = new JTable(new guk.editIM.GlyphTable());
    // see also JTableHeader()

  /**
   * The second table (for many to many compositions, using the data
   * model defined in StringTable.java).
   * @see StringTable
   */
  JTable table2 = new JTable(new guk.editIM.StringTable());

  /**
   * A status bar, and the frame that represents the this-object.
   * Updated by the action listener.
   */
  JLabel statusBar = new JLabel("(status bar)");

  /**
   * The frame that we are.
   */
  JFrame frame;

  /**
   * We collect lists of installed locales and unicode ranges here.
   * Set up by connectIM, used by the look setup and the action
   * listener.
   */
  java.util.List installedLocales = new ArrayList();

  /**
   * Have a global InputMethod control object for virtual keyboard
   * control and maybe other things. Used by the feel parts.
   */
  Object imControl = null;

  /**
   * the subMenuProcessor is used to display some sub menus.
   * if it has done something, it throws an ActionEvent to
   * the assigned listener (given as constructor argument).
   * Contains or calls the help and about window, the file
   * type / name requester, glyph button and glyph palette
   * processing and other functions like smaller dialogues
   * to request values from the user, and the clipboard tools.
   */
  EditIMPopups subMenuProcessor = null;

  /**
   * the fileCommandProcessor is responsible for reading,
   * writing and merging files into the editor. It can pop
   * up confirmation dialogues, but things like selection
   * of file names and file types are done by subMenuProcessor.
   * Knows the MapTable objects to communicate with.
   * When needed, fileCommandProcessor throws an ActionEvent to
   * the assigned listener (given as constructor argument).
   */
   FileCommands fileCommandProcessor = null;

  /**
   * The MenuHelpers class provides functions like creating menu
   * items and buttons. Initially, no action listener is set.
   */
  MenuHelpers menu = new MenuHelpers(null);



  /**
   *  The constructor calls methods to build the look and feel
   *  @param unifont is a Unicode-capable font. If null, a default
   *  font will be used (with the help of the FontLoader).
   *  @param iMDesc an input method descriptor to be used, or null,
   *  which will cause the editor to connect to GUK IM itself.
   *  @param iM an input method object to connect to. If null, the
   *  editor will request one itself.
   *  @param pFrame the frame to which the user should be able to
   *  send key events, using the type to client function. If null,
   *  type to client will not be available.
   */
  public EditIM(Font unifont, InputMethodDescriptor iMDesc,
    InputMethod iM, Frame pFrame) {
    frame = this;
    menu.setListener(this); // items will have this action listener
    menu.setBaseWin(this);  // centering will be relative to this
    menu.setSyncFont(unifont); // syncFont will apply this font
    if (pFrame == null) {
      frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
      // we have no parent frame: just exit completely when exiting
    } else {
      frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      // exit, dispose, do_nothing, ...
    }
    enableEvents(AWTEvent.WINDOW_EVENT_MASK); // why?
    try {
      connectIM(iMDesc, iM); // activate IM connection
      editIMLookInit(unifont); // build GUI
      editIMFeelInit(); // add functionality
      subMenuProcessor.setTypeTarget(pFrame); // for type to client
    } catch (Exception e) {
      DebugEditIM.println(0, "Error setting up menu or function of EditIM:");
      e.printStackTrace();
    }
    frame.validate();
    // could resize or move (frame.setLocation(x,y)) here
    // Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    // Dimension mySize = frame.getSize();
    frame.setLocation(5,20); // top left (window manager buttons above...)
    frame.setVisible(true);
  }// public EditIM()


  /**
   * Init IM connection and Locale list.
   * basically taken from GATE GUK Editor.
   * @param iMDesc An input method descriptor, or null. If null,
   * we will open connection to GUK IM ourselves, and also ignore
   * the iM parameter.
   * @param iM An input method, or null. When null, several features
   * will not be available!
   */
  private void connectIM(InputMethodDescriptor iMDesc, InputMethod iM) {
    InputMethodDescriptor imd = null;
    InputMethod im = null;
    try{
      if (iMDesc != null) { // use caller I.M.D. if any
        imd = iMDesc;
        im = iM;
      } else {
        //if this fails guk is not present
        Class.forName("uk.ac.gate.guk.im.GateIMDescriptor");
          // was: guk.im.GateIMDescriptor
        //add the Gate input methods
        imd = new /* uk.ac.gate.guk.im */ GateIMDescriptor(); /* *** */
        im = imd.createInputMethod(); /* *** */
      }
      installedLocales.addAll(Arrays.asList(imd.getAvailableLocales()));
      imControl = im.getControlObject();
      if ((imControl == null) || !(imControl instanceof GateIM)) {
        imControl = null;
        DebugEditIM.println(0, "Could not get GUK IM control object");
      }
      DebugEditIM.println(0, "GUK IM loaded, " + installedLocales.size()
        + " Locales available");
    }catch(Exception e){
      //something happened; most probably guk not present.
      //just drop it, is not vital.
    }
    Collections.sort(installedLocales, new Comparator(){
      public int compare(Object o1, Object o2){
        return ((Locale)o1).getDisplayName().
	  compareTo(((Locale)o2).getDisplayName());
      }
    });
  }// private void connectIM()


  /**
   * Here we pile up lots of menu items in front of the user.
   * Several items will use the requested font.
   * @param unifont Several items will use this font and assume
   * that it can display all Unicode chars. If null, a default
   * font will be loaded with the help of {@link FontLoader
   * FontLoader}.
   * @throws Exception If anything goes wrong, the Exception
   * will be passed on to the caller.
   */
  private void editIMLookInit(Font unifont) throws Exception {

    /** instance of a simple helper class (only has Iterator getIter())
     * that creates a list of unicode range start points for us. */
    UnicodeRanges unicodeRanges = new UnicodeRanges();
    //
    /** This string will be used to determine button sizes. */
    String longestButtonString = "Language";
    /** The ten hotkeys buttons and the blockIMButton are all
     * member of one group, so they act like radio buttons. */
    ButtonGroup top10Group = new ButtonGroup();
    /** The split pane where the tables are displayed to the user. */
    JSplitPane splitPane = null;
    /** will contain two scrollpanes with textpanes */
    JPanel contentPane;
    /** The menu bar will use a grid layout to save width. */
    JMenuBar menuBar = new JMenuBar();
    /** Menus with Unicode character ranges. The user can select
     *  to hide or show them. Showing one scrolls to it. Three menus
     *  because there are lots of character ranges. */
    JMenu jumpMenu = null;
    JMenu jumpMenu2 = null;
    JMenu jumpMenu3 = null; // in Java 1.4, we have lots of ranges!
    /** The well-known menus for file and help. The file menu
     * also contains options like GUK IM control and locale button
     * assignment. */
    JMenu fileMenu = null; // load/save
    JMenu helpMenu = null; // help and similar stuff
    /** Sub-menus of the file menu. */
    JMenu loadLocaleMenu = null; // load from locale
    JMenu loadButtonMenu = null; // load to button
    /** A toolbar with buttons to select one of 10 GUK locales and the
     * glyph preview button. The locale buttons can be assigned by the
     * user. The glyph preview button shows the currently selected
     * glyph and allows further actions with it. */
    JPanel localeBar = new JPanel();
    Iterator iLIter;
    int i;

    if (unifont == null) {
      unifont = new Font("Arial Unicode MS", // default if null given
        Font.PLAIN, 12);
    }
    menu.setSyncFont(unifont); // syncFont will apply this font
    menu.syncFont(frame);      // use the Unicode font everywhere
    // (might be better for performance to use it only where needed!?)
    if ((imControl != null) && (imControl instanceof GateIM)) {
		((GateIM)imControl).setKeyboardFont(unifont);
		// obviously, the virtual keyboard should be able to
		// display Unicode glyhps, too.
	}

    menuBar.setLayout(new GridLayout(2,0)); // two rows - less width!
      // (default is FlowLayout (wraps), or BoxLayout (single row)
    this.setJMenuBar(menuBar);

    // BUILD MENUBAR

    fileMenu = (JMenu)menu.myJMenuItem("menu", "File", KeyEvent.VK_F,
      "load and save keymaps, control input methods", null);
    menuBar.add(fileMenu);

    jumpMenu = (JMenu)menu.myJMenuItem("menu", "Unicode I", KeyEvent.VK_U,
      "Jump to Unicode locations and "
      + "control their visibility in the table (part 1)", null);
    menuBar.add(jumpMenu);

    jumpMenu2 = (JMenu)menu.myJMenuItem("menu", "Unicode II", KeyEvent.VK_C,
      "Jump to Unicode locations and "
      + "control their visibility in the table (part 2)", null);
    menuBar.add(jumpMenu2);

    jumpMenu3 = (JMenu)menu.myJMenuItem("menu", "Unicode III", KeyEvent.VK_D,
      "Jump to Unicode locations and "
      + "control their visibility in the table (part 3)", null);
    menuBar.add(jumpMenu3);

    jumpField.setText("");
    jumpField.setEditable(true);
    jumpField.setToolTipText("Enter a position like \\u1234 or a "
      + "glyph to jump there in the list");
    menu.syncFont(jumpField); // must be able to show unicode
    jumpField.addActionListener(menu.getListener());
    jumpField.setMinimumSize(new Dimension(
      frame.getFontMetrics(jumpField.getFont()).
        stringWidth("\\u8a8a++"), 0));
      // reserve enough space for typing some unicode glyph escape
    jumpField.setPreferredSize(jumpField.getMinimumSize());
    jumpField.setMaximumSize(new Dimension(
      (int)jumpField.getMinimumSize().getWidth(), Short.MAX_VALUE));

    {
      JLabel scrollLabel = new JLabel("Scroll to: ");
      scrollLabel.setBorder(new EmptyBorder(2,2,2,2));
      scrollLabel.setHorizontalAlignment(JLabel.RIGHT); // to the right
      scrollLabel.setDisplayedMnemonic('t'); // only display, not handle
      scrollLabel.setLabelFor(jumpField); // in combination with
         // setDisplayedMnemonic, also causes mnemonic  key binding!
      menuBar.add(scrollLabel);
      menuBar.add(jumpField); // jump to character / hex position \\u1234
    }

    menuBar.add(new JLabel(" "));
    helpMenu = (JMenu)menu.myJMenuItem("menu", "Help ", KeyEvent.VK_H,
      "find the help window and the about window in this menu", null);
    helpMenu.setHorizontalAlignment(JLabel.RIGHT); // to the (very) right
    menuBar.add(helpMenu);

    // FILE MENU PART 1

    fileMenu.add(menu.myJMenuItem("item", "Load", KeyEvent.VK_L,
      "load a keymap file into the editor", "loadlocale file"));
    fileMenu.add(menu.myJMenuItem("item", "Merge", KeyEvent.VK_M,
      "merge a keymap file into edited one", "mergelocale file"));
    fileMenu.add(menu.myJMenuItem("item", "Save", KeyEvent.VK_S,
      "save the currently edited keymap to a file", "savelocale file"));
    fileMenu.addSeparator();

    // VIRTUAL KEYBOARD MENU

    if (imControl != null) {
      fileMenu.add(menu.myJMenuItem("box", "Show Virtual Keyboard",
        KeyEvent.VK_V,
        "show a virtual keyboard following the real shift states",
        "showvirtualkeyboard tracking"));
      fileMenu.addSeparator();
    }

    // LOCALE BUTTON AND GLYPH PREVIEW BAR

    // localeBar.setFloatable(false); // cannot be detached
    // (if localeBar would be a JToolBar, we setFloatable(false)...)
    /**
     *  Java 1.4.0 / Linux is buggy here: a detached
     *  toolbar is always horizontal, and a re-attached one
     *  is also always horizontal. So detaching is useless for us.
     */
    iLIter = installedLocales.iterator();
    hotkeys = new LocaleHotkey [10];
    int w, h, barW, barH;
    i = 0;
    FontMetrics fm = frame.getFontMetrics(frame.getFont());
    w = fm.stringWidth(longestButtonString);
    h = fm.getHeight();

    localeBar.setLayout(new BoxLayout(localeBar, BoxLayout.Y_AXIS));
    { // limit variable scope
      JPanel localeBGrid1 = new JPanel(new GridLayout(0,1)); // 1 column
      blockIMButton = new JToggleButton();

      JLabel imLabel = new JLabel(" Use IM: ");
      imLabel.setBorder(new EmptyBorder(2,2,2,2));
      imLabel.setAlignmentX(0.5f); // center
      imLabel.setHorizontalAlignment(JLabel.LEFT);
      imLabel.setHorizontalTextPosition(JLabel.LEFT);
      imLabel.setDisplayedMnemonic('i'); // only display, not handle
      imLabel.setLabelFor(blockIMButton); // in combination with
         // setDisplayedMnemonic, also causes mnemonic  key binding!
         // (setLabelFor target cannot be ButtonGroup, must be Component.
         // (for localeBGrid1 is not useful)
      menu.fixSize(imLabel, new Dimension(w+8,h+4));
      localeBar.add(imLabel);

      blockIMButton.setText("none");
      blockIMButton.setToolTipText(
        "use system keyboard map, no Input Method, in text fields");
      blockIMButton.setActionCommand("blockimusage tables");
      blockIMButton.addActionListener(menu.getListener());
      blockIMButton.setMargin(new Insets(2,2,2,2)); // only use a small margin
      blockIMButton.enableInputMethods(false);
      localeBGrid1.add(blockIMButton); // localeBar.add(blockIMButton);

      // hotkeys[0] = new LocaleHotkey(frame.getLocale());
      // hotkeys[0].setText("THIS"); // do not forget to REDO this when needed!
      // add to localebar, i++ // not useful, as current locale might be non-GUK
      // better use "none" to select default (no IM) keymap

      while (iLIter.hasNext() && (i < hotkeys.length)) {
         hotkeys[i] = new LocaleHotkey((Locale)iLIter.next());
         hotkeys[i].enableInputMethods(false);
         localeBGrid1.add(hotkeys[i]); // use a JPanel or a JToolBar?
           // (a JToolBar prefers Actions/icons, but other stuff is okay, too)
         i++;
      }
      for (i=0; (i<hotkeys.length) && (i<10); i++) {
         if (hotkeys[i] == null) {
           DebugEditIM.println(0, "Null hotkey!? Index: " + i);
           continue;
         }
         top10Group.add(hotkeys[i]); // only one of them can be on!
         menu.fixSize(hotkeys[i], new Dimension(w+8,h+4));
         frame.getRootPane().getInputMap().
           put(KeyStroke.getKeyStroke("F"+(i+1)),
             "Hotkey F" + (i+1));
             // use "none" as second object to ignore,
             // use null to remove keystroke binding.
         frame.getRootPane().getActionMap().put("Hotkey F" + (i+1),
           new HotKey(hotkeys[i]));
         hotkeys[i].addActionListener(menu.getListener());
      }
      menu.fixSize(blockIMButton, new Dimension(w+8,h+4));
      top10Group.add(blockIMButton);
      blockIMButton.setSelected(true);

      menu.fixSize(imLabel,  new Dimension(w+8,h+4));
      // update to match size

      localeBGrid1.validate(); // or doLayout() ?
      localeBar.add(localeBGrid1); // grid of none- and normal locale buttons

      barW = (int)(localeBGrid1.getLayout().
        minimumLayoutSize(localeBGrid1).getWidth());
      barH = (int)(localeBGrid1.getLayout().
        minimumLayoutSize(localeBGrid1).getHeight());
    } // limit variable scope

    // GLYPH PREVIEW AND MENU BUTTON

    menu.syncFont(glyphButton);
    glyphButton.setFont(glyphButton.getFont().deriveFont(36.0f));
      // special font: bigger version of the Unicode font.
    updateGlyphButton('a'); // set size and similar stuff
      // The glyph button is initialized THERE !
    glyphButton.addActionListener(menu.getListener());
      // should do some font metrics and maximum char size as
      // above to get other sizes, but I prefer to swim along
      // with other locale button sizes here.
    // localeBar.add(glyphButton);
      // moved into a grid, see right below

    { // limit variable scope
      JPanel localeBGrid2 = new JPanel(new BorderLayout());
        // same width for preview button and field of small buttons
      JPanel glyphBPanel = new JPanel(new GridLayout(1,4));
        // a field of tiny buttons

      localeBar.add(Box.createVerticalGlue());
        // all bonus space in localeBar goes here
      JLabel gbLabel = new JLabel(" Glyph:");
      gbLabel.setAlignmentX(0.5f); // center
      gbLabel.setBorder(new EmptyBorder(2,2,2,2));
      gbLabel.setHorizontalAlignment(JLabel.LEFT);
      gbLabel.setHorizontalTextPosition(JLabel.LEFT);
      gbLabel.setDisplayedMnemonic('g'); // only display, not handle
      gbLabel.setLabelFor(glyphButton); // in combination with
         // setDisplayedMnemonic, also causes mnemonic  key binding!
      menu.fixSize(gbLabel, new Dimension(w+8,h+4));
      localeBar.add(gbLabel);
      barH += h + h + 20; // do not forget the labels!

      glyphBPanel.add(menu.glyphXButton("S", // send to client
        "type the previewed glyph to our client", "GLYtype"));
      glyphBPanel.add(menu.glyphXButton("C",
        "copy the previewed glyph to the clipboard", "GLYcopy"));
      glyphBPanel.add(menu.glyphXButton("P", // paste to lower table
        "paste the previewed glyph to the lower table", "GLYaddcharif"));
      glyphBPanel.add(menu.glyphXButton("J", // jump there in upper table
        "jump to the previewed glyph in the upper table", "GLYjumpfield"));
      //
      glyphBPanel.setAlignmentX(glyphButton.getAlignmentX()); // sigh...
      localeBGrid2.add(glyphButton, BorderLayout.NORTH);
      localeBGrid2.add(Box.createVerticalStrut(3), BorderLayout.CENTER);
        // glue both parts together
      localeBGrid2.add(glyphBPanel, BorderLayout.SOUTH);
        // might have a bigger minimum size than the other parts,
        // but bad luck...
      localeBGrid2.validate();
      menu.fixSize(localeBGrid2, localeBGrid2.getPreferredSize());
      localeBar.add(localeBGrid2); // grid with glyph preview and its buttons

      DebugEditIM.println(3, "only localeBGrid1 and labels: "
        + "barW=" + barW + " barH=" + barH);
      if (((int)(localeBGrid2.getLayout().
        minimumLayoutSize(localeBGrid2).getWidth())) > barW)
        barW = (int)(localeBGrid2.getLayout().
          minimumLayoutSize(localeBGrid2).getWidth());
      barH = barH + 30 + (int)(localeBGrid2.getLayout().
        minimumLayoutSize(localeBGrid2).getHeight());
      DebugEditIM.println(3, "with localeBGrid2 and bonus: "
        + "barW=" + barW + " barH=" + barH);
    } // limit variable scope

    localeBar.validate();
    menu.fixSize(localeBar, new Dimension(barW, barH));
    localeBar.setMaximumSize(new Dimension(barW+25,1000));
        // based on the assumption that our toolbar is vertical!
    localeBar.validate();

    // LOCALE LOAD / SAVE MENUS, BUTTON MENU

    loadLocaleMenu = (JMenu)menu.myJMenuItem("menu", "Activate locale",
      KeyEvent.VK_A,
      "activate an available locale as input locale for this editor", null);
    loadLocaleMenu.add(menu.myJMenuItem("item", "Block IM in text fields",
      KeyEvent.VK_X, "use system keyboard rather than IM in text fields",
      "blockimusage tables"));
    loadLocaleMenu.addSeparator();

    loadButtonMenu = (JMenu)menu.myJMenuItem("menu", "Assign locale IM to button",
      KeyEvent.VK_B, "load a locale input method into a quickselect button",
      null);

    iLIter = installedLocales.iterator();
    while (iLIter.hasNext()) {
      Locale loc = (Locale)iLIter.next();
      String nam = loc.getDisplayName();
      loadLocaleMenu.add(menu.myJMenuItem("item", "Activate: " + nam, -1,
        "activate an existing locale as our input keymap",
        "activatelocale " + loc.toString() ));
      loadButtonMenu.add(menu.myJMenuItem("item", "Assign " + nam + " to button",
        -1, "assign existing locale input method to a quickselect button",
        "setlocalebutton " + loc.toString() ));
    }

    // FILE MENU PART 2

    fileMenu.add(loadLocaleMenu);
    fileMenu.add(loadButtonMenu);
    fileMenu.addSeparator();

    fileMenu.add(menu.myJMenuItem("item", "Clipboard Tools...", KeyEvent.VK_C,
      "open a window with a text field connected to the system clipboard",
      "clipmenu open"));
    fileMenu.addSeparator();

    fileMenu.add(menu.myJMenuItem("item", "Exit", KeyEvent.VK_X,
      "close the locale keymap editor", "exit localeeditor"));
    // ... setAccelerator(KeyStroke.
    //  getKeyStroke(KeyEvent.VK_F4, ActionEvent.ALT_MASK));

    // HELP MENU

    helpMenu.add(menu.myJMenuItem("item", "Help", KeyEvent.VK_H,
      "show a window with information on how to use this software",
      "showwindow help"));
    helpMenu.add(menu.myJMenuItem("item", "About", KeyEvent.VK_A,
      "show a window with information about this software and your system",
      "showwindow about"));

    // BUILD "JUMP TO" TOOLBAR, EHM, MENU (too big for a toolbar)

    jumpMenu.add(menu.myJMenuItem("box",
      "Show USED glyphs", KeyEvent.VK_U,
      "Temporarily hide all glyphs that the edited mapping does not use"
      + " (from the upper table)", "hideunmapped glyphs"));
    // depending on the implementation, this may or may not
    // cause all used glyphs to become visible (the showall / showpart
    // status may still hide them). Current: Show used wins over hide.
    jumpMenu.addSeparator();

    Iterator iter = unicodeRanges.getIter();
    int ranges = 0;
    while (iter.hasNext()) {
      if (Character.UnicodeBlock.of( (char)
        ((Integer)(iter.next())).intValue() ) != null)
      ranges++; // only count useful ranges!
    }
    iter = unicodeRanges.getIter();
    if (iter.hasNext()) {
       int count = 0;
       int start = ((Integer)(iter.next())).intValue();
       int end = start;
       while (iter.hasNext()) {
         start = end; // shift trough
         end = ((Integer)(iter.next())).intValue(); // shift through
           // last entry is defined to be Character.MAX_VALUE, so
           // it does not start a new range. So this works...
         Character.UnicodeBlock ucb =
           Character.UnicodeBlock.of((char)start);
         if (ucb == null) {
           // unused block, skipped for now
         } else { // non-null block
           String itemName = "Show all " + ucb.toString()
             + " [\\u" + menu.toHex(start) + "-\\u"
             + menu.toHex(end-1) + "]";
           JMenuItem menuItem =  menu.myJMenuItem("box", itemName, -1,
             itemName + " and jump there",
             "showall " + start + "-" + (end-1) );
           if ((end - start) <= 0x100) { // small range
             // for a small range, no other menu items are needed
             // as we have only one entry, we can even use the main menu:
             if (count >= (2 * ranges / 3)) {
               jumpMenu3.add(menuItem); // third menu -  avoid too long menu
             } else if (count >= (ranges / 3)) {
               jumpMenu2.add(menuItem); // second menu -  avoid too long menu
             } else {
               jumpMenu.add(menuItem); // standard menu
             }
           } else { // bigger range
             JMenu cSubMenu = (JMenu)menu.myJMenuItem("menu", ucb.toString(), -1,
               "Enable " + ucb.toString() + " glyphs to be listed,"
               + "and jump to the selected place", null);
             int PARTS = 32; // maximum sub area count
             int step = 128; // minimum sub area size
             int mid = start; /* not ... + 0x100 */
             if ((end - start) > (step*PARTS))
               step = (end - start) / PARTS; // menu size sanity limit
             menuItem.setMnemonic(KeyEvent.VK_A);
             cSubMenu.add(menuItem); // item for whole range
             while (mid < end) {
               itemName = "Show glyphs \\u"
                 + menu.toHex(mid) + " to \\u"
                 + menu.toHex(mid+step-1);
               cSubMenu.add(menu.myJMenuItem("box", itemName, -1,
                 itemName + " and jump there",
                 "showpart " + mid + "-" + (mid+step-1) ));
               mid += step;
               // working of the checkboxes: show whole range true
               // overrides subranges. Subranges are unchanged if
               // subrange enable status changes.
             }// while
             if (count >= (2 * ranges / 3)) {
               jumpMenu3.add(cSubMenu); // third menu -  avoid too long menu
             } else if (count >= (ranges / 3)) {
               jumpMenu2.add(cSubMenu); // second menu -  avoid too long menu
             } else {
               jumpMenu.add(cSubMenu); // standard menu
             }
           }// bigger range
         count++;
         }// non-null
       }// while
    }// have unicode ranges (not really needed, but encapsulates variables)

    // ADD BODY PARTS

    contentPane = (JPanel) frame.getContentPane();
    contentPane.setLayout(new BorderLayout());

    frame.setTitle("Input Method Editor");

    JPanel doubleStatus = new JPanel();
    doubleStatus.setLayout(new BorderLayout());
    glyphBar = new JPanel();
    glyphBar.setLayout(new FlowLayout());

    doubleStatus.add(glyphBar, BorderLayout.NORTH);
    doubleStatus.add(statusBar, BorderLayout.SOUTH);

    // NORTH: nothing but the MenuBar
    // EAST:  nothing
    contentPane.add(localeBar, BorderLayout.WEST); /* top 10 buttons */
    contentPane.add(doubleStatus, BorderLayout.SOUTH);

    frame.pack(); // auto-determine minimum dimensions
    contentPane.setMinimumSize(frame.getSize());
    contentPane.setPreferredSize(frame.getSize());
    // Trick: add splitPane AFTER dimension calculations!

    menu.syncFont(table1);
    menu.syncFont(table2);
    // notice that we did not set the font of their JTableHeaders...

    splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
      new JScrollPane(table1, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED),
      new JScrollPane(table2, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED));
      // many to one (keystroke string to unicode) table
      // and many to many (user definable) table
    splitPane.setResizeWeight(0.9); // most changes modify top half size
      // most space for many to one table
    splitPane.setDividerSize(3); // small divider, please
    splitPane.setContinuousLayout(true); // keep redrawing while divider moves
    contentPane.add(splitPane, BorderLayout.CENTER); // chars and strings

    frame.pack(); // update minimum / preferred dimensions
    splitPane.setDividerLocation(
      splitPane.getMinimumDividerLocation() + (int)(0.66 * (
        splitPane.getMaximumDividerLocation() -
        splitPane.getMinimumDividerLocation() )) );
    // splitPane.resetToPreferredSizes(); // auto-weigh splitting

    JScrollPane table1SP = (JScrollPane)splitPane.getTopComponent();
      // scrollpane of upper table, which is table1
    JViewport table1VP = table1SP.getViewport();
    table1VP.setScrollMode(JViewport.BLIT_SCROLL_MODE);
      // modes: BLIT: graphics.copyArea() BACKINGSTORE: lots of RAM...
      //        SIMPLE: redraw whole thing on every move.

    ((GlyphTable)table1.getModel()).setScroller(table1VP, table1, menu);
      // tell this special table model how to scroll itself
      // and which font to use

    glyphButton.enableInputMethods(false);
    menuBar.enableInputMethods(false);

  }// private void editIMLookInit()


/* *********** That was the look, now the feel ************* */


  /**
   * Here we add the FUNCTIONALITY to the menu items,
   * if not done already in editIMLookInit().
   * Also initializes functionality that is provided
   * by separate classes.
   * @throws Exception If anything goes wrong, the Exception
   * will simply be passed on to the caller.
   */
  private void editIMFeelInit() throws Exception {
    int i, j, w, h;
    FontMetrics fm = frame.getFontMetrics(frame.getFont());
    java.awt.geom.Rectangle2D maxCharRect = fm.getMaxCharBounds(
      frame.getContentPane().getGraphics());
    w = (int)maxCharRect.getWidth() + 5;
    // removed the following, because the number is now in the ToolTip:
      // w += fm.stringWidth(" \\ua8a8"); // *** space for glyph number ***
    h = (int)maxCharRect.getHeight() + 5;

    // table init: either use table1. ... directly (viewed coordinates)
    // or better use table1.getModel()....
    table1.setRowHeight(h + table1.getRowMargin());
    table2.setRowHeight(1 * (h + table2.getRowMargin()));

    DebugEditIM.println(1, "Setting up glyph table GUI");

    // the following just bends the settings so that the
    // model can provide the renderer itself and so that
    // the renderer is able to use INTERNAL data.
    // This is probably a *** KLUDGE !?!? ***.
    // The EditorWarper simply copies colors, font and tool tip
    // from the renderer of a cell to the editor of the same cell.
    for (int column = 0; column <= 2; column++) {
      table1.setDefaultRenderer(table1.getColumnClass(column),
        (MapTable)table1.getModel());
      table2.setDefaultRenderer(table2.getColumnClass(column),
        (MapTable)table2.getModel());
      table1.setDefaultEditor(table1.getColumnClass(column),
        new EditorWarper(new JTextField()) );
      table2.setDefaultEditor(table2.getColumnClass(column),
        new EditorWarper(new JTextField()) );
    }

    table1.setSelectionMode(0 /* JList.SINGLE_SELECTION */);
    table2.setSelectionMode(0 /* JList.SINGLE_SELECTION */);
    table1.setRowSelectionAllowed(true);
    table2.setRowSelectionAllowed(true);
      // Cell and Column sel are disabled by default anyway
    table1.getColumnModel().getColumn(0).setMinWidth(w + 4 );
    table1.getColumnModel().getColumn(0).setPreferredWidth(w + 25 );
    table1.getColumnModel().getColumn(0).setMaxWidth(w + 25 );
      // fix to useful width for single glyph
    table2.getColumnModel().getColumn(0).setMinWidth(2 * w + 4 );
    table2.getColumnModel().getColumn(0).setPreferredWidth(2 * w + 25 );
      // recommend useful width for two (or more) glyphs
    table1.getColumnModel().getColumn(1).setMinWidth(2 * w + 4 );
    table1.getColumnModel().getColumn(1).setPreferredWidth(10 * w + 25 );
    table2.getColumnModel().getColumn(1).setMinWidth(2 * w + 4 );
    table2.getColumnModel().getColumn(1).setPreferredWidth(10 * w + 25 );
      // should be enough for a few keys
    table1.getColumnModel().getColumn(2).setMinWidth(10);
    table1.getColumnModel().getColumn(2).setPreferredWidth(5 * w + 25 );
    table2.getColumnModel().getColumn(2).setMinWidth(10);
    table2.getColumnModel().getColumn(2).setPreferredWidth(5 * w + 25 );
      // can be almost nothing

    // initialize submenu processor (font, action listener and
    // centering base defined by properties of the menu object).
    subMenuProcessor = new EditIMPopups(menu);
    subMenuProcessor.setLocaleButtons(hotkeys);
      // help sub menu processor to label choices related to
      // the locale buttons
    subMenuProcessor.setGlyphBar(glyphBar);
      // give the sub menu processor a panel where it can present
      // glyphs and an user interface for using them
    menu.tellListener(subMenuProcessor, "glyphbuttonupdate #");
      // tickle the sub menu processor,
      // so that it sets up the glyph bar GUI

    // initialize the file command processor (which coordinates
    // loading, saving and merging of files from / to the tables).
    // Also uses a Font and a Component in the same way as the
    // submenu processor did.
    fileCommandProcessor = new FileCommands(menu,
      (MapTable)(table1.getModel()), (MapTable)(table2.getModel()));

/**
 * BUG: JScrollPane does not get it, so the H scrollbar stays off :-(
 */
    // BUG - table1.getColumnModel().getColumn(1).setMinWidth(250);
    table1.getColumnModel().getColumn(1).setPreferredWidth(350);
      // recommend bigger width for more text
    // BUG - table2.getColumnModel().getColumn(1).setMinWidth(250);
    table2.getColumnModel().getColumn(1).setPreferredWidth(350);
      // recommend bigger width for more text
/** (BUG end)
 */

    table1.getSelectionModel().
      addListSelectionListener(this);
    table2.getSelectionModel().
      addListSelectionListener(this);
      // only possible when row selection is allowed.
      // triggers valueChanged(ListSelectionEvent e) where we can
      // use e.g. e.getFirstIndex()

   frame.pack();

  }// private void editIMFeelInit()


  /**
   * Shows a new glyph on the glyph button and updates the
   * related action command and tool tip.
   * Helper method to re-set the size of our glpyh button and to
   * give it a new label and action to match the given char.
   * @param glyph The Unicode character which should be associated
   * to the glyph button GUI component.
   */
  void updateGlyphButton(char glyph) {
    FontMetrics fm = glyphButton.getFontMetrics(glyphButton.getFont());
    java.awt.geom.Rectangle2D maxCharRect = fm.getMaxCharBounds(
      frame.getContentPane().getGraphics());
    int w = (int)maxCharRect.getWidth() + 5;
    int h = (int)maxCharRect.getHeight() + 5; // OR: fm.getHeight() + 5;
    glyphChar = glyph; // for other related commands
    glyphButton.setActionCommand("glyphbuttonupdate " + glyph);
      // use glyphbutton if you want a popup menu, and
      // glyphbuttonupdate if you only want to update the history
      // (useful if the history already has its own user interface!)
    glyphButton.setText("" + glyph);
    glyphButton.setMargin(new Insets(2,2,2,2));
    menu.fixSize(glyphButton, new Dimension(w+4,h+4));
    glyphButton.setMaximumSize(new Dimension(w+50,h+50));
    glyphButton.setToolTipText("Add glyph \\u"
      + menu.toHex((int)glyph) + " [" + glyph + "] of "
      + Character.UnicodeBlock.of(glyph) + " to the glyph palette.");
      // (With popup: Add glyph to palette and show more options)
  }// updateGlyphButton


  /**
   * A classical method: set the title depending on whether
   * the edited document is changed (and should be saved before
   * leaving the Input Method Editor).
   */
   public void updateTitle() {
     if ( ((MapTable)(table1.getModel())).isChanged() ||
       ((MapTable)(table2.getModel())).isChanged() ) {
       frame.setTitle("Input Method Editor (changed)");
     } else {
       frame.setTitle("Input Method Editor");
     }
   } // updateTitle


  /**
   * This collects all user activity in form of events,
   * parses it, and distributes it as needed.
   * <p>
   * One actionPerformed method where all the events meet :-)
   * This also dispatches events to other parts of EditIM. It
   * calls actionPerformed rather than processEvent to bypass
   * unnecessary processing. Please READ THE CODE to know which
   * commands are parsed by which of the EditIM classes!
   * </p><p><pre>
   * Things that are handled directly here:
   * addcharif char - append char to table2 selected row, left
   *    column, IF a row is selected. Be careful not to disturb
   *    editing sessions in table2. Use "addchar row char" command
   *    which itself is interpreted by table2/processCommand...
   * activatelocale locale (not file) - activate locale for our IM
   *    (do not edit, just activate it as our InputContext IM).
   *    (also "clicks" the newly assigned first button)
   * selectlocale locale - as activatelocale, but without click
   * setlocalebuttonlocale button locale - just call appropriate
   *    hotkeys[...].loadlocale(Locale).
   * ...
   * Things that use the "checked" boolean value:
   * showvirtualkeyboard tracking - enable/disable normal GUK IM
   *    virtual keyboard (shows the mapping for the current state)
   * showvirtualkeyboard type - enable extended GUK virtual keyboard
   *    of special type (e.g. "what-happens-when Ctrl-Alt-something
   *    is pressed") - NOT IMPLEMENTED (not even in GUK!).
   * blockimusage where - disable usage of IM in our input context
   *    (where value ignored). Only the system keyboard driver is
   *    used in that mode for table and jumpfield editing.
   * exit localeeditor - leave this editor, by dispose or by exit,
   *    depending on whether there is a parent Frame.
   * </pre></p>
   */
  public void actionPerformed(ActionEvent e) {
    // super.actionPerformed(e);
    boolean checked = false;
    String command = null;
    DebugEditIM.println(4, "ActionEvent: " + e);
    updateTitle();
    int mod = e.getModifiers();
    if (e.getID() != ActionEvent.ACTION_PERFORMED) {
      DebugEditIM.println(1, "Other ActionEvent: " + e);
      return;
    }
    if (e.getSource().equals(jumpField)) {
      command = "jumpfield " + e.getActionCommand();
      menu.tellListener(command); // re-dispatch...
      return;
    }
    command = e.getActionCommand();
    if ( command.startsWith("GLY") && (command.indexOf(" ") < 0) ) {
      // add glyphChar to GLY commands if needed
      // which generally means that they are produced here)
      DebugEditIM.println(3, "re-sent with glyph [" + glyphChar
        + "]: " + command);
      menu.tellListener(subMenuProcessor, command + " " + glyphChar);
      return;
    }
    DebugEditIM.println(1, "ActionEvent: <" + command
      + "> [Modifiers=" + e.getModifiers() + "] ");
    statusBar.setText("  action: " + command + " "
      + " [Modifiers=" + e.getModifiers() + "]" );
    statusBar.setFont(frame.getFont());
    //
    try {
      java.lang.reflect.Method [] meth =
        e.getSource().getClass().getMethods();
      for (int i = 0; i < meth.length; i++) {
        if (meth[i].getName().equals("getState")) { // checkbox
          checked = ( (JCheckBoxMenuItem)(e.getSource()) ).getState();
          DebugEditIM.println(2, "getState=" + checked);
          statusBar.setText(statusBar.getText() + " [checked="
            + checked + "]" ); // about e.g. JCheckBoxMenuItems
        } else if (meth[i].getName().equals("isSelected")) { // button
          checked = ( (JCheckBoxMenuItem)(e.getSource()) ).isSelected();
          DebugEditIM.println(2, "isSelected=" + checked);
          statusBar.setText(statusBar.getText() + " [selected="
            + checked + "]" );  // about e.g. JToggleButtons
        } else {
          DebugEditIM.println(4, "has: " + meth[i].getName());
          // LONG list of other methods in this class
        }
      } // for Method array...
    } catch (Exception exc) {
      // we can ask a Method for name, return type, parameter types...
      // not an object that has a checkbox-like state getName()
      // but sometimes security tells us that we are not to know!
    } // catch
    //
    if (command.equals("exit localeeditor")) {

      if (( (MapTable)(table1.getModel()) ).isChanged() ||
          ( (MapTable)(table2.getModel()) ).isChanged()) {
          if (JOptionPane.YES_OPTION !=
            JOptionPane.showConfirmDialog(menu.getBaseWin(),
              "Unsaved changes, exit anyway?",
              "Input Method Editor Warning",
              JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE))
        return; // ignore exit request if user did not confirm
      }

      frame.dispose(); // should probably map this to "frame exit"
      // or the other way round, to have a common "are you sure"
      // dialogue before exiting ***
      if (frame.getDefaultCloseOperation() == JFrame.EXIT_ON_CLOSE)
        System.exit(0);
      // exit completely, if there is no "key receiving parent frame"

    // elsewhere: } else if (command.startsWith("typeclient ")) {

      // ... KeyEvent keyE = AssignObject.typedKeyEvent('x', 0);
      // keyE.setSource(pframe); // target and "source" !
      // pframe.dispatchEvent(keyE);

    } else if (command.startsWith("addcharif ")) {

      char theChar = command.charAt("addcharif ".length());
      // if command ends right after "addcharif ", an exception occurs
      int theRow = table2.getSelectedRow();
      if ((theRow != -1) &&
          ( (table2.getEditingRow() != theRow) ||
            (table2.getEditingColumn() != 0) ) ) {
        table2.getModel().setValueAt(
          table2.getModel().getValueAt(theRow, 0).toString() + theChar,
          theRow, 0); // just append glyhp (could be improved...)
      } else {
        Toolkit.getDefaultToolkit().beep();
        if (theRow == -1) {
          DebugEditIM.println(1,
            "No row selected, no paste target in lower table.");
          menu.tellListener(subMenuProcessor, "errormessage "
            + "You must select something\n"
            + "in the lower table to mark\nthe target row.");
        } else {
          DebugEditIM.println(1,
            "Paste target into lower table cannot bark in yet.");
          menu.tellListener(subMenuProcessor, "errormessage "
            + "You cannot yet paste into\nan ongoing glyphs edit yet.");
        }
      }

    } else if (command.startsWith("setlocalebuttonlocale ")) {

      StringTokenizer strtok = new StringTokenizer(
        command.substring("setlocalebuttonlocale ".length()) );
      int index = -1;
      try {
        index = Integer.parseInt(strtok.nextToken());
        String localeS = strtok.nextToken(""); // all the rest
        Iterator iLIter = installedLocales.iterator();
        boolean found = false;
        while (iLIter.hasNext()) {
          Locale loc = (Locale)iLIter.next();
          if (loc.toString().trim().equals(localeS.trim())) {
            // found locale by toString name
            hotkeys[index].loadLocale(loc); // arm button with locale!
            DebugEditIM.println(1, "Assigning locale <"
              + loc.toString() + "> to button " + index);
            found = true;
          } // if found
        } // while more locales
        if (!found) {
          DebugEditIM.println(0, "Did not find locale matching <"
            + localeS + "> while trying to assign it to button " + index);
        } // if not found at all
      } catch (Exception ee) {
        DebugEditIM.println(0, "Error trying to update locale buttons!");
        ee.printStackTrace();
      } // catch

    } else if ( command.startsWith("selectlocale ")
      || command.startsWith("activatelocale ") ) {

          String localeS = command.startsWith("select") ?
            command.substring("selectlocale ".length()).trim() :
            command.substring("activatelocale ".length()).trim();
          //
          menu.tellListener("enableimusage tables");
            // "send" yourself the command to re-enable IM usage...
          //
          InputContext ic = frame.getInputContext();
          Iterator iLIter = installedLocales.iterator();
          Locale myLocale = null;
          Locale gukLocale = (imControl == null) ? null :
            ((InputMethod)imControl).getLocale();
          //
          DebugEditIM.println(1, "    Locale of IC of Frame: "
            + ((ic == null) ? "none" : ("" + ic.getLocale()))
            + ", Locale of GUK IM: "
            + ((gukLocale == null) ? "n/v" : gukLocale.toString())
          );
          // frame.getLocale is not interesting (just for i18n)
          // ... neither is frame.setLocale() for us now ...
          //
          while (iLIter.hasNext()) {
            myLocale = (Locale)iLIter.next();
            if ( myLocale.toString().trim().equals(localeS) ) {
              if (ic.selectInputMethod(myLocale)) { // worked
                DebugEditIM.println(1, "    IC Locale set to: <"
                  + ic.getLocale() + ">, Frame Locale is: <"
                  + frame.getLocale() + ">");
              } else { // failed
                DebugEditIM.println(1, "    IC selectInputMethod for "
                  + "locale <" + myLocale + "> failed.");
                // could do ... .setSelected(false) here...
                return;
              } // failed
              if (command.startsWith("activate")) {
                hotkeys[0].loadLocale(myLocale); // arm button with locale
                hotkeys[0].doClick(); // select this one
              } // activating
              break; // leave while
            } // if string matches
          } // while

    } else if ( command.startsWith("blockimusage ") ||
                command.startsWith("enableimusage ") ) {

      // arguments are ignored for now, just control all text fields.
      // (we have turned IM off for all primary menus, anyway)

      checked = command.startsWith("enable");
      table1.enableInputMethods( checked );
      table2.enableInputMethods( checked );
      // make selection affect the CURRENT editing session, too! (if any)
      if (table1.getEditorComponent() != null)
         table1.getEditorComponent().enableInputMethods( checked );
      if (table2.getEditorComponent() != null)
         table2.getEditorComponent().enableInputMethods( checked );
      jumpField.enableInputMethods( checked );
      if (!e.getSource().equals(blockIMButton)) { // update from menu:
        blockIMButton.setSelected(!checked);
      }
      DebugEditIM.println(1,"Input Method usage for editing text fields "
        + "turned " + ( checked ? "ON (default)"
          : "OFF (use system keyboard driver only)" ));

    } else if (command.startsWith("showvirtualkeyboard ")) {

      String whichVK =
        command.substring("showvirtualkeyboard ".length()).trim();
      if (whichVK.equals("tracking")) {
        Object imObject = getInputContext().getInputMethodControlObject();
        if(imObject != null && imObject instanceof GateIM){
          ((GateIM)imObject).setMapVisible( checked );
          DebugEditIM.println(1, "IC virtual keyboard configured");
        } else if (imControl != null) {
          // now trying keyboard map of local IM:
          ((GateIM)imControl).setMapVisible( checked );
          DebugEditIM.println(1, "local virtual keyboard configured");
        } else {
          DebugEditIM.println(0, "No virtual keyboards available!");
        }
      } else {
        DebugEditIM.println(0,
          "virtual keyboard type not yet supported: " + whichVK);
        // would parse this and use for setMapVisibleX( ... )
        // (for future GUK IM versions)
      }

    } else { // other commands: forward to other processors!

      ( (GlyphTable)(table1.getModel()) ).processCommand(
        command, e.getModifiers(), checked);
        /**
         * <p><pre>
         * --- table1 processes: ---
         * jumpto glyphnumber / jumpfield glyph-or-\\uxxx
         * showall firstglyphnumber lastglyphnumber (using checked)
         * showpart firstglyphnumber lastglyphnumber (using checked)
         * </pre></p>
         */

      ( (StringTable)(table2.getModel()) ).processCommand(
        command, e.getModifiers(), checked);
        /**
         * table2 only processes: addchar row char
         */

      menu.tellListener(subMenuProcessor, e); // or postEvent()?
        /**
         * <p><pre>
         * --- subMenuProcessor processes: ---
         *    (can throw back any of the OTHER commands)
         * glyphbutton CHAR
         * glyphbuttonupdate CHAR
         * glyphbarbutton CHAR (internal use)
         * glyphbarcombo updated (internal use)
         * GLYtype CHAR (internal and external use)
         * GLYcopy CHAR (internal and external use)
         * GLYjumpfield CHAR (internal and external use)
         * GLYaddcharif CHAR (internal and external use)
         * loadlocale file / mergelocale file / savelocale file
         * setlocalebutton locale
         * showwindow help / showwindow helpdone
         * showwindow about / showwindow aboutdone
         * clipmenu open
         * errormessage text
         * typeclient glyph
         * </pre></p>
         */

      menu.tellListener(fileCommandProcessor, e); // or postEvent()?
        /**
         * The file I/O component should process those commands:
         * loadlocalefile encoding filetype filename - load for editing!
         *   ... similar to: ...
         * mergelocalefile encoding filetype filename - ADD to edited!
         * savelocalefile encoding filetype filename - create file.
         */

      // needed after at least file load / save / merge and
      // jumpto, jumpfield, addcharif, showall, showpart...:
      table1.repaint();
      table2.repaint();

      // if some area just got enabled, make scrolling to that
      // point easy. Cannot scroll directly: the table may still
      // be busy with reorganizing itself....
      if ( ( command.startsWith("showall ") ||
        command.startsWith("showpart ") ) && checked ) {
	    StringTokenizer strtok =
	      new StringTokenizer(command, " -");
	    if (strtok.countTokens() >= 2) {
  	      /* 1st word = */ strtok.nextToken();
  		  // menu.tellListener("jumpto " + strtok.nextToken());
  		  jumpField.setText(strtok.nextToken().trim());
	    } // was: "showall 12-34" with checked on
      }


    } // other commands
  }// actionPerformed


  /**
   * If the user has selected some table1 row, update the glyph
   * button. If the user has selected the last table2 row, add
   * a new row to table2.
   * Triggered by things for which we are the ListSelectionListener.
   */
  public void valueChanged(ListSelectionEvent e) {
    if (e == null) return;
    // e.getSource() does not get us far: just some instance of
    // DefaultListSelectionModel...
    int row = table1.getSelectedRow(); // in visible rows
    int row1 = e.getFirstIndex(); // in visible rows
    int row2 = e.getLastIndex(); // in visible rows
    boolean adj = e.getValueIsAdjusting();
    //
    int tab2row = table2.getSelectedRow() + 1;
    updateTitle();
    if (tab2row >= table2.getRowCount()) { // last row selected?
      ((StringTable)table2.getModel()).addRow(); // then add a row!
    } // last row of table 2 selected
    //
    DebugEditIM.println(3, "rows " + row1 + " to " + row2
      + " selected, " + (adj ? "adjusting" : "settled")
      + ", table reports selected row "
      + ((row >= 0) ? ("" + row) : "none") );
    if ((row1 < 0) || (row2 < 0) || (row < 0) || adj)
      return; // (user UNselected sth., or boring "adjusting" event)
      // strange: mouse and key down select first=new, last=old,
      // mouse also has an "adjusting" event, BUT key up selects
      // first=old last=new ---> must use table1.getSelectedRow()!
    updateGlyphButton(
      table1.getModel().getValueAt(row, 0).toString().charAt(0)
    ); // configure glyph preview and action button for this char
    return;
  }// valueChanged


  /**
   * Overridden so that we can exit when window is closed.
   * @param e A window event, telling us about closing,
   * iconifying, activation, similar things.
   */
  protected void processWindowEvent(WindowEvent e) {
    if (e.getID() == WindowEvent.WINDOW_CLOSING) {
      DebugEditIM.println(2,"About to close (maybe after confirm)");
      menu.tellListener("exit localeeditor");
      // skip further processing of the closing event!
    } else {
      super.processWindowEvent(e);
    }
  }// processWindowEvent(WindowEvent e)


 /**
  * main - to allow running of the Input Method Editor as standalone.
  */
 public static void main(String[] args) {

   // InputMethodDescriptor imd = null;
   // InputMethod im = null;
   // Object imControl = null;

   try {
     UIManager.setLookAndFeel(
       UIManager.getSystemLookAndFeelClassName());
   } catch(Exception e) {
     e.printStackTrace();
   } // try / catch

   Font theFont = FontOptimizer.unicodeFont();

   DebugEditIM.println(0,
    "Input Method Editor running in standalone mode");
   new EditIM(theFont /* null for default */,
     null /* no other imd */, null /* no other im */,
     null /* no frame to send keys to */);

   // because fontLoader (etc.) lives longer than the frame,
   // we would not exit after closing the frame normally. BUT as
   // we have no "frame to send keys to", closing is mapped to EXIT.

 } // main


}// class JFrame


/////////////////////////////////////////////////////////////////


/**
 * Inner class: factory to assign target.doClick() to an Action.
 * Not really used / usefull currently.
 */
class HotKey extends AbstractAction {
  AbstractButton target;

  /**
   * create a default abstract action and remember the target.
   */
  public HotKey(AbstractButton o) {
    super(); // create an AbstractAction with default text and icon
    target = o;
  }

  /**
   * Click the target when anything happens.
   */
  public void actionPerformed(ActionEvent e) {
    DebugEditIM.println(2, "ActionEvent: " + e);
    target.doClick();
  }

}// class HotKey

