/**
 * Helper for the EditIM Input Method Editor:
 * displays some sub menus like file requesters and choice
 * popups and similar stuff. Throws results as ActionEvents.
 * Delegates tasks to other classes.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*; // Clipboard ...
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.beans.*;
import java.io.*;
// import guk.editIM.MenuHelpers;
// import guk.editIM.DebugEditIM;
// import guk.editIM.FileRequester;
// import guk.editIM.HelpEditIM;
// import guk.editIM.ClipWindow;


/**
 * Helper for the EditIM Input Method EditorI:
 * displays some sub menus like file requesters and choice
 * popups and similar stuff. Throws results as ActionEvents.
 * Delegates tasks to other classes.
 */
public class EditIMPopups implements ActionListener {


 /**
  * The MenuHelpers class provides functions like creating menu items
  * and buttons. Initially, no action listener is set.
  */
 MenuHelpers menu = new MenuHelpers(null);


 /**
  * A Frame to where the type to client function applies
  * (for the current implementation, can be any Component)
  */
 Frame pframe = null;


 /**
  * An instance of an helper class which will provide the
  * help and about popup windows.
  */
 HelpEditIM help = null;


 /**
  * An instance of an helper class which will provide the file
  * requester (with special features for the input method editor).
  */
 FileRequester fileReq = null;


 /**
  * An instance of an helper class which will provide an
  * utility window to use the system clipboard more easily.
  */
 ClipWindow clipper = null;


 /**
  * An array of things that might help to label our buttons
  * for the locale button related choice popups
  */
 Object [] localeButtons = null;


 /**
  * The history Vector for the glyph button and the dialog for that
  * plus the combobox used for the history and a JPanel to show
  * buttons and combo boxes which also hold the history
  */
 static final int glyphPalSize = 30;
 java.util.Vector glyphHistory = new java.util.Vector(glyphPalSize + 1);
 JFrame glyphDialog = new JFrame("Glyph paste");
 JComboBox glyphHistoryCombo = new JComboBox();
 JPanel glyphHistoryBar = null;
 JComboBox glyphModeCombo = null;


 /**
  * The constructor needs a MenuHelpers instance for generic
  * settings like font, action listener and centering base.
  * Instantiates the other helper classes.
  * @param menuHelper Provides functions like creating menu items.
  * Should have the listener and font set.
  */
 public EditIMPopups(MenuHelpers menuHelper) {
   menu = menuHelper;
   help = new HelpEditIM(menu);
   fileReq = new FileRequester(menu);
   clipper = new ClipWindow(menu);
 } // constructor


 /* *** *** */


 /**
  *  Popup a menu that allows to copy/add/type (to clipboard,
  *  table2 or client respectively) the current glyph or one
  *  from a glyph history. only pops up when popup is true,
  *  just adds glyph to history else.
  *  @param glyph The glyph for which the menu is to be presented
  *  and which should be added to the history / palette area.
  *  The history forgets the oldest entry when it runs out of space.
  *  @param popup Only when this is true, a menu will pop up.
  *  Otherwise, only the history / palette is updated.
  */
 void glyphPopup(char glyph, boolean popup) {
   if (!glyphHistory.contains(new Character(glyph))) {
     if (glyphHistory.size() >= glyphPalSize) // full?
       glyphHistory.removeElementAt(0); // remove first entry
     glyphHistory.add(new Character(glyph)); // add to history
   } else {
     // was already in history, could re-sort history here
   }
   //
   if (glyphHistoryBar != null) { // present BONUS user interface
     // glyphHistoryBar.setLayout(new BoxLayout(glyphHistoryBar,
     //   BoxLayout.X_AXIS));
     glyphHistoryBar.setLayout(new BorderLayout());
     if (glyphModeCombo == null) {
       Object [] glyphModes = { "clipboard",
         "scroll target", "lower table", "client frame" };
       glyphModeCombo = new JComboBox(glyphModes);
       glyphModeCombo.setActionCommand("glpyhbarcombo updated");
       glyphModeCombo.setSelectedIndex(2);
       glyphModeCombo.setEditable(false);
       glyphModeCombo.setToolTipText("select what should be updated"
         + " when you click one of the glyph palette items");
       glyphModeCombo.addActionListener(menu.getListener());
       glyphModeCombo.setAlignmentY(0.0f); // drift up
     }
     glyphHistoryBar.removeAll();
     JPanel glyphGrid = new JPanel(new GridLayout(3,0)); // 3 rows
     JPanel comboPanel = new JPanel();
     comboPanel.setLayout(new BoxLayout(comboPanel, BoxLayout.X_AXIS));
       // do not use default (FlowLayout, which wraps) unless you can
       // resize the glyphHistoryBar in hight as reaction to width resizes
     {
       JLabel gMCLabel = new JLabel(
         " Glyph palette sends to: ");
       gMCLabel.setBorder(new EmptyBorder(2,2,2,2));
       gMCLabel.setVerticalTextPosition(JLabel.TOP);
       gMCLabel.setVerticalAlignment(JLabel.TOP);
       gMCLabel.setAlignmentY(0.0f);
       gMCLabel.setDisplayedMnemonic('s'); // only display, not handle
       gMCLabel.setLabelFor(glyphModeCombo); // in combination with
         // setDisplayedMnemonic, also causes mnemonic  key binding!
       comboPanel.add(gMCLabel);
     }
     comboPanel.add(glyphModeCombo);
     comboPanel.add(Box.createHorizontalGlue());

     JButton oneGlyph = null;
     java.util.Iterator histIter = glyphHistory.iterator();
     int i = 0;
     while (histIter.hasNext()) {
       char ch = ( (Character)(histIter.next()) ).charValue();
       String hexTex = Integer.toHexString((int)ch);
       while (hexTex.length() < 4)
         hexTex = "0" + hexTex; // padding
       oneGlyph = menu.glyphXButton("" + ch,
         "click to activate glyph " + "(\\u" + hexTex + ")",
         "glyphbarbutton " + ch);
       {
         menu.syncFont(oneGlyph);
         java.awt.geom.Rectangle2D maxCharRect =
           glyphGrid.getFontMetrics(oneGlyph.getFont()).getMaxCharBounds(
           glyphGrid.getGraphics());
         menu.fixSize(oneGlyph, new Dimension(
           (int)maxCharRect.getWidth() + 6,
           (int)maxCharRect.getHeight() + 6));
       }
       glyphGrid.add(oneGlyph);
       i++;
     } // while
     while (i < glyphPalSize) { // fill rest
       oneGlyph = menu.glyphXButton("*",
         "this is an unused glyph button",
         "glyphbarunusedbutton " + i);
       oneGlyph.setEnabled(false);
       glyphGrid.add(oneGlyph);
       i++;
     } // while
     glyphHistoryBar.add(comboPanel, BorderLayout.NORTH);
     // ... Box.createHorizontalGlue() ...
     glyphHistoryBar.add(glyphGrid, BorderLayout.CENTER);
     glyphHistoryBar.add(Box.createHorizontalGlue(), BorderLayout.EAST);
   } // bonus user interface
   //
   if (!popup) return; // just updated the history
   //
   glyphHistoryCombo = new JComboBox(); // re-init
   JPanel glyphBar = new JPanel();
   if (glyphDialog != null)
     glyphDialog.dispose(); // remove old instance!
   glyphDialog = new JFrame("Glyph paste"); // re-init
   JPanel glyphDialogPane = (JPanel)glyphDialog.getContentPane();
   glyphDialogPane.setLayout(new BorderLayout());
   menu.syncFont(glyphDialogPane);
   glyphHistoryCombo.setFont(glyphDialogPane.getFont().deriveFont(24.0f));
     // bigger font for the history thing
   glyphBar.setLayout(new BoxLayout(glyphBar,
     BoxLayout.Y_AXIS)); // vertical layout
   //
   java.util.Iterator histIter = glyphHistory.iterator();
   while (histIter.hasNext()) {
     glyphHistoryCombo.addItem(""
       + ( (Character)(histIter.next()) ).charValue() );
   }
   glyphHistoryCombo.setSelectedItem("" + glyph);
   glyphHistoryCombo.setEditable(false);
   glyphHistoryCombo.setMaximumSize(glyphHistoryCombo.getPreferredSize());
   glyphHistoryCombo.setActionCommand("GLYcombo");
   //
   glyphBar.add(menu.myJButton("to clipboard", KeyEvent.VK_C,
     "copy selected glyph to system clipboard", "GLYcopy"));
     // to be exact: default clipboard. Should be the system clipboard.
   glyphBar.add(
     Box.createHorizontalGlue());
   glyphBar.add(menu.myJButton("scroll to", KeyEvent.VK_S,
     "scroll upper table to selected glyph", "GLYjumpfield"));
   glyphBar.add(menu.myJButton("to lower table", KeyEvent.VK_T,
     "add selected glyph to selected lower table row glyphs",
     "GLYaddcharif"));
   if (pframe != null)
     glyphBar.add(menu.myJButton("type to client", KeyEvent.VK_P,
       "simulate typing selected glyph to client", "GLYtype"));
     // typing to null client makes no sense.
   glyphBar.add(menu.myJButton("just remember", KeyEvent.VK_R,
     "keep this glyph in history but cancel other actions with it",
     "GLYcancel"));
   //
   glyphDialogPane.add(glyphHistoryCombo, BorderLayout.WEST);
   glyphDialogPane.add(glyphBar, BorderLayout.EAST);
   glyphDialog.pack(); // optimize size
   menu.centerMe(glyphDialog);
   glyphDialog.show();
   //
 } // glyphPopup


 /**
  * Postprocess glyph dialog results and create commands in
  * proper syntax for our listener, or take action directly.
  * The jumpfield and addcharif commands are passed on,
  * combo is ignored, cancel closes the glyph menu popup.
  * The type to client command sends the glyph to the pFrame
  * client in form of a KeyEvent, and the copy command tries
  * to copy the glyph to the clipboard, which usually crashes
  * in Java 1.3.1 Unix, so be careful.
  * @param type The kind of action that should be taken, in
  * form of a string (jumpfield, addcharif, type, copy, combo
  * or cancel).
  * @param glyph The character to which the command should
  * apply, unless the type string already specifies one.
  */
 void glyphListener (String type, char glyph) {
   String command = null;
   if (type.equals("addcharif") || type.equals("jumpfield")) {
     //
     command = type; // just pass on to other processor
     //
   } else if (type.equals("type")) {
     //
     // now handled directly here, so command is just to inform people.
     command = "typeclient";
     if (pframe != null) {
       DebugEditIM.println(2, "type [" + glyph + "] to Frame <"
         + pframe.getTitle() + "> (" + pframe.getName() + ")");
       try {
         KeyEvent keyE =
           AssignObject.typedKeyEvent(glyph, 0 /* modifiers */);
           // cannot adjust time and id. leaving type at KEY_TYPED.
           // leaving keycode, modifiers and keychar as is.
         keyE.setSource(pframe);
           // will be processed by listeners belonging to pframe and
           // finally by pframe itself: we pretend it to be FROM there.
           // could call pframe.getFocusOwner() to know more. This
           // may return null. Could do pframe.requestFocus().
         pframe.dispatchEvent(keyE);
           // send Event to the client! Will cause processEvent there.
       } catch (Throwable th) {
         DebugEditIM.println(1,"type to client failed: " + th);
       } // catch
     } // pframe exists
     else {
       DebugEditIM.println(0, "cannot type [" + glyph + "] to null Frame");
       menu.tellListener("errormessage "
         + " no client frame active to which you could \"type\".");
     } // no pframe
     //
   } else if (type.equals("copy")) {
     //
     /** ... use system clipboard in more elegant way? */
     java.awt.Toolkit.getDefaultToolkit().getSystemClipboard().
       setContents( new StringSelection("" + glyph),
         null /* no owner */ );
       // an owner would be notified if the clipboard is overwritten
       // see Clipboard class: we can also getContents() from it,
       // receiving a Transferable... then, do try/catch Throwable: x =
       // (String) transferable.getTransferData(DataFlavor.stringFlavor)
     DebugEditIM.println(2,
       "copy to (only) to system clipboard");
     command = "addtoclipboard"; // this tells nobody yet :-(
       // WOULD also add to local Java clipboard if we knew how
     //
   } else if (type.startsWith("combo")) {
     //
     return; // user made a selection. nice to know.
     //
   } else if (type.equals("cancel")) {
     //
     glyphDialog.dispose();
     return; // keep history up to date but cancel other dialog actions
     //
   } else {
     //
     DebugEditIM.println(0, "confused by: " + type);
     command = "glyphproblem" + type;
     return; // do not send an action event
     //
   }
   menu.tellListener(command + " " + glyph);
   glyphDialog.dispose(); // free resources again
   DebugEditIM.println(2, "finished: " + command + " " + glyph);
 } // glyphListener


 /**
  * Hand over a panel where we can present glyph related
  * interaction components.
  * @param glyphBar The area where an extended glyph GUI
  * can be displayed (normally no popup) by the glyph
  * interaction components above.
  */
 public void setGlyphBar(JPanel glyphBar) {
   glyphHistoryBar = glyphBar;
 } // setLocaleButtons


 /**
  * Select a Frame (currently, a Component would be enough) to which
  * it sends simulated keypresses (used in the glyph menu).
  * @param pFrame A frame to which, by means of posting KeyEvents,
  * this object should be able to send simulated key presses. Used
  * for the type to client function, which will be unavailable when
  * no pFrame is set.
  */
 public void setTypeTarget(Frame pFrame) {
   pframe = pFrame;
 } // setTypeTarget


 /* *** *** */


 /**
  * Builds a popup where the user can select a locale button
  * slot to trigger a setlocalebuttonlocale button locale command.
  * Triggers a selectlocalebutton button locale command, where
  * button is the button number selected by the user (the popup
  * shows some user friendly combo box) and locale is localeS.
  * The actual GUI part of the work is done in queryLocaleButton.
  * @param localeS The locale name to be appended to the command.
  */
 void assignLocaleButton(String localeS) {
   Object [] content = {
     "Please select a button to which the locale",
     "<" + localeS + "> should be assigned"
   }; // can be any number of Objects (preferrably Strings)
   int result = queryLocaleButton(content);
   if (result < 0) return; // user did not select anything
   DebugEditIM.println(1, "localeS=" + localeS
     + " => User selected button=" + result );
   menu.tellListener("setlocalebuttonlocale " + result + " " + localeS);
 } // assignLocaleButton


 /**
  * Generic "ask to select a locale button" helper.
  * Implements the GUI described at assignLocaleButton.
  * @returns -1 or the index of the selected locale button.
  */
 int queryLocaleButton(Object [] question) {
   int max = 10;
   if (localeButtons != null) {
     max = localeButtons.length;
     if (max < 2) return max-1;
     // return the only possible selection directly
     if (max > 26) max = 26; // limit to a) .. z)
   }
   Object [] values = new Object[max];
   for (int i = 0; i < max; i++) {
     values[i] = "" + ((char)('a' + i)) + ") "; // as an ID code
     if (localeButtons == null) {
       values[i] = values[i] + "Button " + (i+1);
     } else if (localeButtons[i] instanceof LocaleHotkey) {
       values[i] = values[i] + "["
         + ((LocaleHotkey)localeButtons[i]).askLocale().getLanguage()
         + "] <"
         + ((LocaleHotkey)localeButtons[i]).askLocale().getDisplayName()
         + ">";
       // use getLanguage() to get 2 letter code,
       // use toString() to get something like "en__FOO BAR"
     } else if (localeButtons[i] instanceof JButton) {
       values[i] = values[i] + ((JButton)localeButtons[i]).getText();
     } else {
       values[i] = values[i] + localeButtons[i].toString();
     }
   }
   String result = (String)JOptionPane.showInputDialog(
        menu.getBaseWin() /* parent Component */,
        question, "Which button?" /* title */,
        JOptionPane.QUESTION_MESSAGE /* selects icon */,
        null /* no user icon */, values /* choice */,
        values[0] /* initial choice */);
   if ( (result == null) || result.equals("Cancel")
     || (result.length() < 1) )
     return -1; // user did not answer
   return (int)(result.charAt(0) - 'a');
   // return user selection as a number from 0 to (max-1)
 } // assignLocaleButton


 /**
  * Hand over buttons to us, to ease labelling the choice items
  * in every locale button related popup menu.
  * @param buttons Some array of buttons or other objects. Their
  * names or whatever seems to properly describe them are used
  * to fill the choice menu that allows the user to select a
  * button.
  */
 public void setLocaleButtons(Object [] buttons) {
   localeButtons = buttons;
 } // setLocaleButtons


 /* *** *** */


 /**
  * The main point where events are coming in.
  * <p>
  * this is an ActionListener, so it must handle actionPerformed. All
  * requests are coming in through this method. Each type of request
  * should be handled by calling one or more of the private methods
  * of this class. This is the common parser for all. Some of the
  * methods above throw events back by using menu.tellListener().
  * </p>
  * <p><pre>
  * Accepted commands:
  * glyphbutton CHAR - popup menu with history selection and
  *    buttons to copy or add to table2 or type to client
  * glyphbuttonupdate CHAR - just update the history selection
  *    (which has an additional non-popup user interface in current
  *    versions, so the popup menu mode is no longer needed)
  * glyphbarbutton CHAR - perform action as selected by glyphModeCombo
  * glpyhbarcombo updated - glyphModeCombo value has changed
  * GLYtype CHAR - send char as key event to our pframe
  * GLYcopy CHAR - send char to clipboard
  * GLYjumpfield CHAR - redirected to jumpfield CHAR (scroll to...)
  * GLYaddcharif CHAR - redirected to addcharif CHAR (add to table)
  *   (all GLY... commands are also internally used by the glyph popup)
  * loadlocale file - ask for a filename AND type to be loaded
  * mergelocale file - as loadlocale file, but for merge.
  * savelocale file - ask for filename AND type
  * setlocalebutton locale - ask for a button number
  * showwindow help - show help window (JTextArea/...)
  * showwindow about - show about window and some properties
  *    does not need to show lots of status (hard to implement)
  * clipmenu open - show window with system clipboard interface
  * errormessage text - show text as an error message popup
  * typeclient glyph - pretend that somebody typed the glyph
  *    into the window of our creator! (currently only triggered
  *    by glyphbutton processing and also DONE right there).
  * </pre></p>
  */
 public void actionPerformed(ActionEvent e) {
   String command = e.getActionCommand();
   int modifiers = e.getModifiers();
   // Object source = e.getSource();
   if (command.startsWith("GLY")) { // internal glyph dialog even!
     char glyph;
     if ((glyphHistoryCombo.getSelectedItem() != null) &&
         glyphDialog.isShowing()) {
       glyph = ((String)(glyphHistoryCombo.getSelectedItem())).charAt(0);

     } else {
       glyph = command.charAt(command.length()-1);
       command = command.substring(0, command.length()-1).trim();
     }
     glyphListener(command.substring(3), glyph); // strip "GLY"
     // redirect to the special listener
     return; // do not process event further
   } // GLY...
   else if (command.startsWith("CLIP")) { // internal clipboard dialog even!
     clipper.clipCommand(command.substring(4)); // strip "CLIP" and
     // redirect to the special listener / processor
     return; // do not process event further
   } // CLIP...
   else if (command.startsWith("errormessage ")) { // error message
     JOptionPane.showMessageDialog(menu.getBaseWin(),
       command.substring("errormessage ".length()),
       "Input Method Editor Error", JOptionPane.ERROR_MESSAGE );
     return; // do not process event further
   } // errormessage
   java.util.StringTokenizer strtok =
     new java.util.StringTokenizer(command);
   try {
     String activity = strtok.nextToken();
     if (activity.equals("glyphbutton")) {
       char glyph = strtok.nextToken().charAt(0);
       DebugEditIM.println(2, "glyphbutton popup requested!");
       glyphPopup(glyph,true);
         // glyphbutton CHAR - popup menu with history selection and
         // buttons to copy or add to table2 or type to client
     } // glyphutton
     else if (activity.equals("glyphbuttonupdate")) {
       char glyph = strtok.nextToken().charAt(0);
       DebugEditIM.println(2, "glyphbutton popup update!");
       glyphPopup(glyph, false);
       // add char to history, do not pop up
     } // glyphuttonupdate
     else if (activity.equals("glyphbarbutton")) {
       char glyph = strtok.nextToken().charAt(0);
       DebugEditIM.println(2, "a glyphbarbutton got clicked!");
       String glyphMode = "" + glyphModeCombo.getSelectedItem();
       if (glyphMode == null) return;
       glyphListener(
         ((glyphMode.indexOf("clipboard") >= 0) ? "copy"
         : ((glyphMode.indexOf("client") >= 0) ? "type"
           : ((glyphMode.indexOf("lower") >= 0) ? "addcharif"
             : "jumpfield") ) ), glyph );
       // trigger activity as selected by glyphModeCombo
     } // glyphuttonupdate
     else if (activity.equals("clipmenu")) {
       DebugEditIM.println(2, "system clipboard popup requested!");
       menu.tellListener("CLIPopen"); // just translate and resend
     } // clipmenu
     else if ( activity.equals("loadlocale")  ||
               activity.equals("mergelocale") ||
               activity.equals("savelocale")  ) {
       menu.tellListener(fileReq, e);  // just pass on to our helper!
     } // loadlocale, savelocale, mergelocale
     else if (activity.equals("setlocalebutton")) {

       String localeS = strtok.nextToken();
       if (localeS.equals("file")) {
         DebugEditIM.println(0, "file (name, type) for assignment "
           + "to button (number) popup requested! NOT IMPLEMENTED!");
       } else {
         DebugEditIM.println(2, "select button number for locale "
           + "assign popup requested!");
         localeS = command.substring("setlocalebutton".length());
         localeS = localeS.trim(); // remove surrounding whitespace
         assignLocaleButton(localeS);
           // builds a popup where the user can select a locale button
           // slot to trigger a setlocalebuttonlocale button locale.
       }
       // just hand over button NUMBER and (fileNAME and TYPE) or
       // LOCALE (just pass through) to help our caller to create
       // a new ActionCommand for one of the locale buttons...

     } // setlocalebutton (2x)
     else if (activity.equals("showwindow")) {
       String winSelect = strtok.nextToken();
       boolean opening = winSelect.indexOf("done") < 0;
       if (winSelect.startsWith("help")) {
         DebugEditIM.println(2,
           "help window " + (opening ? "open" : "clos") + "ing");
         help.helpWindow(opening);
       } else if (winSelect.startsWith("about")) {
         DebugEditIM.println(2,
           "about window " + (opening ? "open" : "clos") + "ing");
         help.aboutWindow(opening);
       }
     } // showwindow (2x)
  } catch (Exception ex) {
    DebugEditIM.println(0, "Unparseable popup command: " + command);
    // ex.printStackTrace();
  }
 } // actionPerformed


} // class EditIMPopups

