/**
 * Helper for the EditIM GUK input method editing GUI:
 * Responsible for reading, writing and merging files
 * into / from the editor tables. It can pop up confirmation
 * dialogues, but things like selection of file names and
 * file types are done by another component.
 * Knows the MapTable objects to communicate with.
 * When needed, fileCommandProcessor throws an ActionEvent to
 * the assigned listener (given as constructor argument).
 * Receives commands by being an ActionListener through
 * actionPerformed, which is triggered after postEvent on us.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.beans.*;
import java.io.*;
import java.util.StringTokenizer;
import java.util.Iterator;
import java.util.Vector;
import java.util.Enumeration;
import java.net.URL;
import java.net.MalformedURLException;

// import guk.editIM.MenuHelpers;
// import guk.editIM.MapTable;
// import guk.editIM.AssignObject;


/**
 * Helper for the EditIM Input Method Editor:
 * Responsible for reading, writing and merging files
 * into / from the editor tables. It can pop up confirmation
 * dialogues, but things like selection of file names and
 * file types are done by another component.
 * Knows the MapTable objects to communicate with.
 * When needed, fileCommandProcessor throws an ActionEvent to
 * the assigned listener (given as constructor argument).
 * Receives commands by being an ActionListener through
 * actionPerformed, which is triggered after postEvent on us.
 */
public class FileCommands implements ActionListener {


 /**
  * The MenuHelpers class provides functions like creating menu
  * items and buttons. Initially, no action listener is set.
  */
 MenuHelpers menu = new MenuHelpers(null);


 /**
  * two tables, one for at most one key sequence for each glyph
  * and one where we can produce several glyphs by one key
  * sequence or assign several key sequences to the same glyphs.
  */
 MapTable tableShort = null;
 MapTable tableLong = null;


 /**
  * The locale name and variant, taken from the inputmethod header of GIM.
  */
 String localeName = "NewLocale";
 String localeNameVariant = "Standard";

 /**
  * All headers that are recognized as such: Lines beginning with option
  * in the case of GIM. No other formats have real headers. Yudit has
  * a format variant marked by a special line (the additive format which
  * &quot:multiplies out&quot; several sub-keymaps), but the Input
  * Method Editor does not handle that one anyway.
  */
 String headers = "";

 /**
  * All comments in the header area. Everything up to the first
  * parseable assignment. Useful for all formats that allow comments
  * or unparseable lines.
  */
 String preComments = "";

 /**
  * All comments after the first parseable assignment that do not
  * belong to an assignment themselves. An alternative would be to
  * Add standalone comments to the next or previous assignment
  * rather than keeping them in one big inComments String.
  */
 String inComments = "";


 /**
  * The constructor initializes important variables, mostly from
  * passed arguments.
  * the constructor needs an ActionListener where all results
  * will be sent to, and an Unicode capable font. In addition,
  * it needs a Component relative to which popups will be places,
  * and two MapTable objects that allow storing and editing of
  * keymap data!
  * @param menuHelper
  * @param glyphTable A MapTable object which is first tried as
  * data source or destination. Half of the editor window.
  * @param stringTable Elements which do not fit glyphTable are
  * put into this stringTable. When we export to a file, first
  * the glyphTable and then the stringTable elements are written,
  * unless some kind of reordering overrides this (may happen for
  * some export file formats).
  */
 public FileCommands(MenuHelpers menuHelper,
    MapTable glyphTable, MapTable stringTable) {
   menu = menuHelper;
   tableShort = glyphTable;
   tableLong = stringTable;
 } // constructor


 /**
  * Shows a probably long log text in a popup window with a
  * given title.
  * @param title The title of the window
  * @param text The content of the window.
  * @param okay If false, the icon will be that of a warning
  * instead of that of an information.
  */
 public void showLog(String title, String text, boolean okay) {
   JOptionPane.showMessageDialog(
     menu.getBaseWin(), new JScrollPane(new JTextArea(text)),
     title, (okay ? JOptionPane.INFORMATION_MESSAGE :
       JOptionPane.WARNING_MESSAGE) );
   // this dialog should be limited to sane size, but the
   // default is quite generous (at most the whole screen).
 } // showLog


 /**
  * Describes an AssignObject data type.
  * @param type An AssignObject type constant.
  * @return A short name for the selected type.
  */
 static public String nameAOType(int type) {
   switch (type) {
     case AssignObject.GIM_FILE: { return "Gate GIM"; }
     case AssignObject.XGIM_FILE: { return "Gate XGIM"; }
     case AssignObject.YUDIT_FILE: { return "Yudit KMAP"; }
     case AssignObject.U8_FILE: { return "MPI U8"; }
     case AssignObject.UNICODE_HUMAN: { return "plain, UTF-8"; }
     case AssignObject.ASCII_HUMAN: { return "plain, ASCII"; }
     case AssignObject.UNICODE_HTML: { return "HTML, UTF-8"; }
     case AssignObject.ASCII_HTML: { return "HTML, ASCII"; }
   } // switch
   return "unknown type";
 } // nameOfType


 /* *** *** */


  /**
   * Load or merge a file into the tables.
   * @param encoding The encoding, typically ISO-8859-1 or UTF-8
   * (both are implemented on all Java platforms).
   * @param filetype The file type code as used by the
   * {@link AssignObject AssignObject} parsing system.
   * @param filename The name of the file to be loaded / merged.
   * @param merge True for merge mode, false for load mode.
   * Load will flush the tables before importing a file.
   * @return False if a really fatal error occurred (true if
   * only warnings occurred or if everything went right).
   */
  public boolean loadFile(String encoding, int filetype,
    String filename, boolean merge) {
    boolean worked = true;
      // false if really troubling error found
    String title = "Import file ("
      + (merge ? "merge" : "load") + ") status report:";
    String troubles = null;
    BufferedReader bufferedReader = null;
    //
    if (!merge) { // load mode is more destructive
      if ((!tableShort.flushTable()) || (!tableLong.flushTable())) {
        troubles += // remember to complain, but continue
          "The tables could not be flushed before loading\n";
      } else {
        DebugEditIM.println(1,
          "Flushed all table data. Now loading new file.");
      }
    }
    if (!merge) {
      headers = "";
      inComments = "";
      preComments = "";
      localeName = "Unknown";
      localeNameVariant = "(no inputmethod header)";
      DebugEditIM.println(1,
        "Flushed all comments, header and locale name information.");
    } // flush metadata, too

    try {
      URL where = new URL(filename);
      bufferedReader = new BufferedReader( // has readLine()
        new InputStreamReader( // allows selecting an encoding
          new BufferedInputStream( // improves performance
            where.openStream()), // fetches the URL
          encoding));
    } catch (UnsupportedEncodingException uee) {
      troubles = "Encoding <" + encoding + "> not supported";
    } catch (MalformedURLException mue) {
      troubles = "Malformed file URL:\n" + filename;
    } catch (IOException ioe) {
      troubles = "File not found or not readable:\n" + filename;
    } catch (Exception e) {
      troubles = "Unknown error: " + e;
    }
    if ((bufferedReader == null) || (troubles != null)) {
      DebugEditIM.println(0,
        "Error opening input file:\n" + troubles);
      JOptionPane.showMessageDialog(menu.getBaseWin(),
        "Could not open file!\n" + troubles, "File Import Error",
        JOptionPane.ERROR_MESSAGE);
      return false; // did not work at all
    }

    troubles = "*** " + title + " ***\n";

    try {
      String line;
      String comment;
      boolean dataStarted = false;
      while ( ( (line = bufferedReader.readLine() ) != null) &&
        worked ) {
        //
        line.trim(); // all formats should allow this
        comment = null;
        if (line.length() < 2) { // too empty
          comment = ""; // also a comment, kind of
        } // empty
        //
        // check for real comment-only lines:
        if (((filetype == AssignObject.GIM_FILE) ||
          (filetype == AssignObject.XGIM_FILE)) &&
          (line.indexOf("#") == 0)) {
          comment = line.substring(1).trim();
        } /* GIM comments (line starts with #) */
        //
        if ( ( (filetype == AssignObject.YUDIT_FILE) ||
               (filetype == AssignObject.GIM_FILE) ||
               (filetype == AssignObject.XGIM_FILE) ) &&
          (line.indexOf("//") == 0)) {
          comment = line.substring(2).trim();
        } /* YUDIT comments (Yudit also accepts // elsewhere on the line */
          /* (Gate also accepts those in GIM files: line starts with //) */
        if (comment != null) {
          if (comment.length() > 0) {
            if (dataStarted) {
              inComments += comment + "\n";
            } else {
              preComments += comment + "\n";
            }
            troubles +=
              (dataStarted ? "BodyComment" : "HeadComment")
              + ": " + comment + "\n";
          } // non-empty comment
        } // found comments
        //
        else if ( ( (filetype == AssignObject.GIM_FILE) ||
          (filetype == AssignObject.XGIM_FILE) ) &&
          line.startsWith("option") ) {
          headers += line + "\n";
          troubles += "*** Header: " + line + "\n";
        } // headers
        //
        else if ( ( (filetype == AssignObject.GIM_FILE) ||
          (filetype == AssignObject.XGIM_FILE) ) &&
          line.startsWith("inputmethod") ) {
          /**
           * Gate actually IGNORES those in GIM files and
           *  only uses im.list for the locale names anyway!
           *  Syntax: inputmethod "language" "variant name"
           */
          StringTokenizer strtok = new StringTokenizer(
            line.substring("inputmethod".length()).trim(), "\"");
            // Maybe TODO: we do not care for escaped quotes here!
          String part1 = null;
          String part2 = null;
          if (strtok.hasMoreTokens()) {
            part1 = strtok.nextToken();
            if (strtok.hasMoreTokens()) {
              /* in between */ strtok.nextToken();
              if (strtok.hasMoreTokens()) {
                part2 = strtok.nextToken(); // (ignore the rest)
              }
            }
          }
          localeName = part1;
          localeNameVariant = part2;
          /** even on MERGE we OVERWRITE the locale name! */
          troubles += "*** Internal GIM locale name (ignored by GUK):\n"
            + "> inputmethod  \"" + part1 + "\"  \"" + part2 + "\"\n";
        } // name giving header
        //
        else try { // real content lines
          //
          AssignObject ao = new AssignObject();
          /** GIM files might allow end of line comments,
           *  so we parse them. GUK cannot (yet) parse them.
           *  So we convert them into comments on their own line
           *  on SAVE (mentioning the "bind" value for reference).
           */
          //
          if ((filetype == AssignObject.YUDIT_FILE) &&
            (!dataStarted) &&
            (line.indexOf("+") > line.indexOf("\"")) ) {
            DebugEditIM.println(0,
              "Unsupported Yudit multi section file!");
            troubles += "*** Unsupported Yudit multi section file!\n";
            // Actually, we would check if the + is BETWEEN the first
            // two quotation marks to be more exact and less paranoid.
            worked = false;
            continue;
          }
          if ( ( (filetype == AssignObject.GIM_FILE) ||
            (filetype == AssignObject.XGIM_FILE) ) &&
            (line.indexOf("resetorsend") >= 0)) {
            troubles += "*** resetorsend translated to simple send "
              + "(as GUK treats it anyway):\n> " + line + "\n";
            preComments += "*** resetorsend translated to send by "
              + "Input Method Editor:\n" + line + "\n";
            int point = line.indexOf("resetorsend");
            line = line.substring(0, point)
              + line.substring(point + "resetor".length());
          } // resetorsend translation
          //
          if ( ( (filetype == AssignObject.GIM_FILE) ||
            (filetype == AssignObject.XGIM_FILE) ) &&
            (line.indexOf("digit") >= 0)) {
            int point = line.indexOf("digit");
            String digits = line.substring(point + "digit".length()).trim();
            /** Optional TODO: we do NOT treat comments in those lines */
            if (digits.indexOf(" ") > 0) {
              // there are actually TWO definitions in this digit line
              /** GIM files would define a toggle hotkey to allow the user
               *  to select either half of the digits dynamically.
               *  We, however, produce two lines in a fixed manner.
               */
              AssignObject ao2 = new AssignObject();
              ao2.importString(line.substring(0, point) + "digit "
                + digits.substring(digits.indexOf(" ")).trim(),
                filetype);
                // second half of that the digit alternative
              ao2.importKeys("C-d" + ao2.exportKeys(filetype), filetype);
                /** Use Control d prefix rather than using a toggle! */
              if ( (!tableShort.addEntry(ao2)) && (!tableLong.addEntry(ao2)) )
                troubles += "*** Could not store digit half: "
                  + ao.exportString(AssignObject.UNICODE_HUMAN) + "\n";
              inComments += "*** digit alternative line split by "
                + "Input Method Editor:\n" + line + "\n";
              line = line.substring(0, point) + "digit "
                + digits.substring(0, digits.indexOf(" ")).trim();
              troubles += "*** Split digit alternative into two assignments:\n"
                + "> " + ao2.exportString(filetype) + "\n> " + line + "\n";
              // leave over first half of that digit alternative
            } // two digits used
          } // digit splitting
          //
          /** TODO: split U8 multiple choice lines into multiple   *
           *  lines (use a loop: if still spaces following, loop,  *
           *  go to the single assignment ao otherwise...)         */
          //
          ao.importString(line, filetype);
          // Idea: try to avoid merge of two identical assignments
          // which would cause a needless choicelist on export and
          // waste space in the lower table...
          if ( (!tableShort.addEntry(ao)) && (!tableLong.addEntry(ao)) )
            troubles += "*** Could not store: "
              + ao.exportString(AssignObject.UNICODE_HUMAN) + "\n";
          dataStarted = true; // inside data area now
        } catch (UnsupportedOperationException uoe) {
          troubles += "*** Parse error (line skipped!): " + uoe.getMessage()
            + "\n> skipped offending line: " + line + "\n";
          // should bail out on certain errors:
          // - Yudit file found being a multi-section one
          // - user selected really unsupported encoding
          // - ???
          // (set worked to false to bail out)
        } // not a comment, try / catch
      } // while not end of file
      //
      if (!dataStarted) {
        troubles += "*** No content found, wrong file type?\n";
        worked = false;
      } // no data at all
      //
    } catch (IOException ioe) {
      troubles += "*** ERROR READING FILE, ABORTED.\n";
      // worked stays true anyway.
    } catch (Exception e) {
      troubles += "*** Internal error while importing:\n" + e;
      e.printStackTrace();
      worked = false;
    } // catch

    troubles += "\nDone with loading file:\n" + filename
      + "\nEncoding: " + encoding + " Type: " + filetype
      + " (" + nameAOType(filetype) + ")\n"
      + (worked ? "(okay)" : "FAILED!") + "\n";

    showLog(title, troubles, worked);
    // Could show preComments, inComments and headers here,
    // but we have already embedded them in the logs anyway.
    tableLong.clearChanged();
    tableShort.clearChanged();
    return worked;
  } // loadFile


 /* *** *** */


  /**
   * Save the tables into a file.
   * @param encoding The encoding, typically ISO-8859-1 or UTF-8
   * (both are implemented on all Java platforms).
   * @param filetype The file type code as used by the
   * {@link AssignObject AssignObject} parsing system.
   * @param filename The name of the file to be loaded / merged.
   * @return False if a really fatal error occurred (true if
   * only warnings occurred or if everything went right).
   */
  public boolean saveFile(String encoding, int filetype,
    String filename) {
    Iterator firstTable = tableShort.readTable().iterator();
    Iterator secondTable = tableLong.readTable().iterator();
    boolean worked = true;
    String title = "Export file status report:";
    String troubles = null;
    StringTokenizer strtok = null;
    String line = null;
    String comment = "// "; // how to start a comment
    BufferedWriter outFile = null;
    try {
      URL where = new URL(filename);
      DebugEditIM.println(1,"Writing file: " + where.getFile());
      OutputStreamWriter oSW = new OutputStreamWriter(
        new FileOutputStream( where.getFile() ),
        encoding);
      outFile = new BufferedWriter(oSW); // optional step.
        // means calling the encoding conversion less often
    } catch (UnsupportedEncodingException uee) {
      troubles = "Encoding <" + encoding + "> not supported";
    } catch (MalformedURLException mue) {
      troubles = "Malformed file URL:\n" + filename;
    } catch (IOException ioe) {
      troubles = "Error writing the file.";
    } catch (Exception e) {
      troubles = "Unknown error: " + e;
    }
    if ((outFile == null) || (troubles != null)) {
      DebugEditIM.println(0,
        "Error opening output file:\n" + troubles);
      JOptionPane.showMessageDialog(menu.getBaseWin(),
        "Could not write file!\n" + troubles, "File Export Error",
        JOptionPane.ERROR_MESSAGE);
      return false; // did not work at all
    }

    troubles = "*** " + title + " ***\n";

 try {

    if ((filetype != AssignObject.GIM_FILE) &&
        (filetype != AssignObject.XGIM_FILE)) { // non-GIM types
     //
      JOptionPane.showMessageDialog(menu.getBaseWin(),
        "This keymap belongs to the locale:\n" + localeName
          + "\nVariant: " + localeNameVariant,
        "Input Method Editor Information",
        JOptionPane.INFORMATION_MESSAGE );
      // other file formats have no locale name header.
      if (filetype != AssignObject.U8_FILE) { // HTML, HUMAN or YUDIT:
        // export the locale name and the headers as comments in
        // classic C++ comment syntax (which Yudit uses)
        outFile.write(comment + "{GIMi] " + "inputmethod \""
          + localeName + "\" \"" + localeNameVariant + "\"\n");
        strtok = new StringTokenizer(headers,"\n");
        while (strtok.hasMoreTokens()) {
          outFile.write(comment + "[GIMo] " + strtok.nextToken() + "\n");
        }
        strtok = new StringTokenizer(preComments,"\n");
        while (strtok.hasMoreTokens()) {
          outFile.write(comment + "[Head] " + strtok.nextToken() + "\n");
        }
        /**
         *  TODO: could try to spread body comments back to their
         *  original places between content lines...
         */
        strtok = new StringTokenizer(inComments,"\n");
        while (strtok.hasMoreTokens()) {
          outFile.write(comment + "[Body] " + strtok.nextToken() + "\n");
        }
      } else { // U8:
        troubles += "*** U8 file. Could not write the following headers:\n"
          + headers + "\n*** U8 file. Could not write the header comments:\n"
          + preComments + "\n*** U8 file. Could not write body comments:\n"
          + inComments + "\n*** U8 file. Could not write locale name:\n"
          + localeName + ", Variant: " + localeNameVariant + "\n";
      }
      //
    } else { // GIM file family
      //
      String newLocaleName = "?";
      comment = "# "; // change comment style
      newLocaleName = JOptionPane.showInputDialog(
        menu.getBaseWin(), "Please give the locale a new name:\n"
        + "Example: Persian, CRL Special Phonetic\n"
        + "(the comma separates name and variant)\n"
        + "Current value:\n" + localeName + ", " + localeNameVariant,
        "Input Method Editor Question", JOptionPane.QUESTION_MESSAGE);
      if (newLocaleName.indexOf(",") >= 0) {
        localeName =
          newLocaleName.substring(0,newLocaleName.indexOf(",")).trim();
        localeNameVariant =
          newLocaleName.substring(newLocaleName.indexOf(",")+1).trim();
      } else {
        DebugEditIM.println(1, "User did not enter a new locale name.");
        // ignore if user has clicked cancel or flunked the syntax
      }

      strtok = new StringTokenizer(preComments,"\n");
      while (strtok.hasMoreTokens()) {
        outFile.write(comment + strtok.nextToken() + "\n");
      }
      outFile.write("inputmethod \""
        + localeName + "\" \"" + localeNameVariant + "\"");
      troubles += ("Wrote GIM locale name: inputmethod \""
        + localeName + "\" \"" + localeNameVariant + "\"");
      strtok = new StringTokenizer(headers, "\n");
      while (strtok.hasMoreTokens()) {
        String token = strtok.nextToken().trim();
        if (token.startsWith("option")) {
          outFile.write(token + "\n");
          troubles += "Restored GIM option: " + token;
        } else {
          outFile.write(comment + token + "\n");
          // a NON GIM header will become a comment when exporting to GIM
        }
      } // while
      strtok = new StringTokenizer(inComments,"\n");
      while (strtok.hasMoreTokens()) {
        outFile.write(comment + "[Body] " + strtok.nextToken() + "\n");
      }
    } // GIM, XGIM

    Vector allEntries = new Vector();
    allEntries.addAll(tableShort.readTable());
    allEntries.addAll(tableLong.readTable());
    /**
     * TODO: To export U8 in a compact way, we have to sort this vector
     * by key sequences and then write the output in a pooled way.
     */

    Enumeration entries = allEntries.elements();
    while (entries.hasMoreElements()) {
      Object entry = entries.nextElement();
      if ((entry == null) || (!(entry instanceof AssignObject))) {
        troubles += "Skipping nonstandard element: " + entry + "\n";
        continue;
      }
      AssignObject ao = (AssignObject)entry;
      if (ao.getKeys().isEmpty() || (ao.getGlyphs().length() == 0)) {
        if (ao.getComments().length() > 0) {
          if (filetype != AssignObject.U8_FILE) {
            outFile.write(comment + "[NONE] " + ao.getComments() + "\n");
          } else {
            troubles += "Lost comment: " + ao.getComments() + "\n";
          }
        } // if comment only element
      } // no assignment
      else { // assignment
        try {
          String oldComments = ao.getComments();
          if (filetype == AssignObject.U8_FILE) {
            if (oldComments.trim().length() > 0)
              troubles += "Lost comments: " + oldComments + "\n";
            ao.setComments("");
          }
          if ((filetype == AssignObject.GIM_FILE) ||
            (filetype == AssignObject.XGIM_FILE)) {
            ao.setComments("");
          }
          outFile.write(ao.exportString(filetype) + "\n");
          //
          if ((filetype == AssignObject.GIM_FILE) ||
            (filetype == AssignObject.XGIM_FILE)) {
            if (oldComments.trim().length() > 0)
              outFile.write(comment + oldComments + "\n");
              // moving comment onto separate line right after the data
          }
          ao.setComments(oldComments); // restore them
        } catch (IOException ioe) {
          troubles += "*** Write error writing object: " + ao + "\n";
        } catch (UnsupportedOperationException uoe) {
          troubles += "*** Conversion error (line skipped!): "
            + uoe.getMessage()
            + "\n> skipped offending object: " + ao + "\n";
        } // entry, try / catch
      } // assignment
    } // while

    outFile.flush();
    outFile.close();

 } catch (IOException ioe) {
   troubles += "*** File write error, aborted! ***\n";
   worked = false;
 }
    troubles += (worked ? "(okay)\n" : "FAILED\n");

    showLog(title, troubles.toString(), worked);
    if (worked) {
      // should be false whenever AssignObjects
      // could not be exported, or a file error occurred.
      tableShort.clearChanged();
      tableLong.clearChanged();
    }
    return worked;
  } // saveFile


 /* *** *** */


 /**
  * <p>
  * This is an ActionListener, so it must handle actionPerformed. All
  * requests are coming in through this method. Each type of request
  * should be handled by calling one or more of the private methods
  * of this class. This is the common parser for all. Some of the
  * methods above throw events back at myListener, in turn.
  * </p>
  * <p><pre>
  * Accepted commands:
  * loadlocalefile encoding filetype filename - load for editing!
  *    ... similar to: ...
  * mergelocalefile encoding filetype filename - ADD to edited!
  * savelocalefile encoding filetype filename - create file.
  *    headers, footers and comments should be preserved from most
  *    recently loaded / merged file if possible!
  * </pre></p>
  */
 public void actionPerformed(ActionEvent e) {
   String command = e.getActionCommand();
   int modifiers = e.getModifiers();
   // Object source = e.getSource();
   String verb = null;
   String encoding = null;
   int filetype = -1;
   String filename = null;
   //
   if (!( command.startsWith("loadlocalefile ")  ||
          command.startsWith("mergelocalefile ") ||
          command.startsWith("savelocalefile ")  )) {
     return; // none of our commands
   } // gate keeper
   try {
     boolean worked = true;
     java.util.StringTokenizer strtok =
       new java.util.StringTokenizer(command);
     verb = strtok.nextToken();
     verb = verb.substring(0, verb.indexOf("localefile"));
       // only leave "verb" part, "load", "merge" or "save".
     encoding = strtok.nextToken(); // e.g. "UTF-8" or "ISO-8859-1"
     filetype = Integer.parseInt(strtok.nextToken()); // type code
     filename = strtok.nextToken("").trim(); // all the rest
       // normally an URL where special chars are escaped.
       // normally begins with file:// for that reason.
     if (verb.equals("save")) {
       /**
        * We expect the file overwrite confirmation and the
        * file writeability check to have already taken place.
        * We COULD, however, ask for confirmation if comments
        * are found and the user tries to save to U8. But we
        * assume that the user knows what (s)he is doing.
        */
       worked = saveFile(encoding, filetype, filename); // save
     } else {
       if (tableShort.isChanged() || tableLong.isChanged()) {
         int confirmed = JOptionPane.showConfirmDialog(
           menu.getBaseWin(), "Unsaved changes, " + verb
           + " file anyway?", "Input Method Editor Warning",
           JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
         if (confirmed != JOptionPane.YES_OPTION) {
           DebugEditIM.println(1, verb + " aborted by user, "
             + "as there were unsaved changes.");
           return; // do what the user wanted and abort loading!
         }
       } // unsaved changes confirmation
       if (verb.equals("load")) {
         worked = loadFile(encoding, filetype, filename, false); // load
       } else {
         worked = loadFile(encoding, filetype, filename, true); // merge
       }
     } // load or merge
     if (!worked)
       JOptionPane.showMessageDialog(menu.getBaseWin(),
         "Failed to " + verb + " file\n" + filename
         + "\nEncoding: " + encoding + ", Type " + filetype + " ("
         + nameAOType(filetype) + ")",
         "Input Method Editor Error", JOptionPane.ERROR_MESSAGE );
     return;
   } catch (Exception ex) {
     DebugEditIM.println(0, "Unparseable file command: " + command);
     // ex.printStackTrace();
     return;
   }
 } // actionPerformed


} // class FileCommands

