/**
 * Helper for the EditIM Input Method Editor:
 * The file requester (for load, save and merge), with a
 * bonus combobox to select the actual file format.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*; // Clipboard ...
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.beans.*;
import java.io.*;
// import guk.editIM.MenuHelpers;
// import guk.editIM.DebugEditIM;


/**
 * Helper for the EditIM Input Method Editor:
 * The file requester (for load, save and merge), with a
 * bonus combobox to select the actual file format.
 */
public class FileRequester implements ActionListener {


 /**
  * The MenuHelpers class provides functions like creating menu items
  * and buttons. Initially, no action listener is set.
  */
 MenuHelpers menu = new MenuHelpers(null);


 /**
  * One choser that is not re-initialiyzed: allows all file open
  * dialogues to remember the CWD from each other etc.
  */
 JFileChooser fileChooser = null;


 /**
  * The constructor needs a MenuHelpers instance for generic
  * settings like font, action listener and centering base.
  * @param menuHelper Provides functions like creating menu items.
  * Should have the listener set.
  */
 public FileRequester(MenuHelpers menuHelper) {
   menu = menuHelper;
 } // constructor


 /* *** *** */


 /**
  * Sends command encoding filetype filename
  * after querying for a file and type
  * (the type will also determine the encoding).
  * This shows a popup with a file choser. A combobox to
  * select some file type is added, and a file filter for
  * the extensions gim, kmap and u8 is provided in addition
  * to the default any file filter. The file type selection
  * also selects the encoding: ISO-8859-1 normally, but
  * UTF-8 for U8 and Unicode simple text files. Yudit is UTF-8
  * for the comments, but the data only uses ASCII characters.
  * The encoding
  * is added to the command as string, the type is added as
  * a string representing one of the
  * {@link AssignObject AssignObject} format constants.
  * The selected file will not be opened, but the name of
  * it will be appended to the command in form of an URL.
  * @param title The title that the dialog window should use.
  * @param command The command that should be PRODUCED. Only
  * the beginning is read to determine the mode of this dialog.
  * Recognized modes are load, save, and merge.
  */
 public void fileTypeRequest(String title, String command) {
   Object [] typeNames = {
     "simple Unicode (UTF-8) text",
     "simple text",
     "standard GUK IM (*.gim)",
     "GIM plus \\k+...+ keystrokes",
     "generic / MPI IM u8 (UTF-8) (*.u8)",
     "Yudit keymap definition (*.kmap)"
   };
   int [] typeCodes = {
     AssignObject.UNICODE_HUMAN,
     AssignObject.ASCII_HUMAN,
     AssignObject.GIM_FILE,
     AssignObject.XGIM_FILE,
     AssignObject.U8_FILE,
     AssignObject.YUDIT_FILE
   };
   String verb = (command.startsWith("save") ? "saved"
     : (command.startsWith("merge") ? "merged" : "loaded"));
   JPanel typePanel = new JPanel();
   JComboBox fileTypeComboBox = new JComboBox( typeNames );
   //
   typePanel.setLayout(new GridLayout(0,1));
     // grid layout 0, 1 means 1 column and rows as needed
     // all components in the grid have the same size,
     // a mix between preferred size and size to make the
     // whole grid fill the current size
   typePanel.add(new JLabel("Please select the content type"));
   typePanel.add(new JLabel("of the file to be " + verb + "."));
   if (verb.equals("saved")) {
     typePanel.add(new JLabel("This will determine the format"));
     typePanel.add(new JLabel("in which data is written."));
   } else {
     typePanel.add(new JLabel("This will select the used parser."));
   }
   typePanel.add(new JLabel("Recommended are GIM and Yudit."));
   if (verb.equals("saved"))
     typePanel.add(new JLabel("Saving to U8 will loose comments!"));
   //
   fileTypeComboBox.setEditable(false);
   fileTypeComboBox.setToolTipText("select a file format"
     + " (recommended: GIM, Yudit)");
   fileTypeComboBox.setSelectedIndex(2); // default: GIM
   typePanel.add(fileTypeComboBox);
   // typePanel.setMaximumSize(new Dimension(300, 400));
   // typePanel.setPreferredSize(new Dimension(300, 400));
     // typePanel.validate();
     // the chooser seems to use only the preferred size
   //
   if (fileChooser == null) { // needs to initialize?
     fileChooser = new JFileChooser("./"); // start in CWD
     try {
       DebugEditIM.println(1, "file requester starting in "
         + fileChooser.getCurrentDirectory().toURL().toString());
       // using URLs because they escape non-ASCII
     } catch (java.net.MalformedURLException mue) {
       DebugEditIM.println(0,
         "file requester initialized, but strange CWD.");
     } // catch
   }
   fileChooser.enableInputMethods(false); // default is true
     // no matter how silly our current IM is, we want to be able
     // to use the keyboard in the file dialog!
   JPanel typePanel2 = new JPanel(new BorderLayout());
   typePanel2.add(typePanel, BorderLayout.SOUTH);
     // putting the panel into another panel, to allow it being
     // smaller than the size of the accessory if it wants to.
   fileChooser.setAccessory(typePanel2);
     // add type selection menu!
   fileChooser.setMultiSelectionEnabled(false);
   fileChooser.setDialogTitle("Select file and type to be " + verb);
   fileChooser.setSelectedFile(null);
     // fileChooser.setCurrentDirectory(File ...)
   fileChooser.setDialogType(
     verb.equals("saved") ? JFileChooser.SAVE_DIALOG
     : JFileChooser.OPEN_DIALOG); // not too important (accessibility)
   fileChooser.setApproveButtonText( // setDialogType also sets this!
     ( verb.equals("saved") ? "Save" :
       (verb.equals("loaded") ? "Load" : "Merge") ));
   // fileChooser.setApproveButtonMnemonic( verb.charAt(0) );
     // not working: defaults for dialog type always win!?
   fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
   /**
    * BONUS PROBLEM: you cannot stop users misusing the chooser
    * to RENAME files, quite strange! Happens even on Unix!
    */
   //
   fileChooser.resetChoosableFileFilters(); // sic!
   fileChooser.setFileFilter(fileChooser.getAcceptAllFileFilter());
   fileChooser.addChoosableFileFilter(
     new javax.swing.filechooser.FileFilter() {
       public String getDescription() {
         return "Keymap files (.gim, .u8, .kmap)";
       }
       public boolean accept(File f) {
         String extS = f.getName();
         if (f.isDirectory())
           return true; // otherwise, directories are invisible...
         if ( (extS.lastIndexOf('.') < 0) ||
           (extS.lastIndexOf('.') == (extS.length()-1)) )
           return false;
         extS = extS.substring(extS.lastIndexOf('.')+1).toLowerCase();
         if (extS.equals("gim") || extS.equals("kmap") || extS.equals("u8"))
           return true;
         return false;
       }
     } // unnamed FileFilter class
       // (PS: we have TWO types called FileFilter, one io and one swing)
   );
   //
   int result = 0;
   do {
     result = fileChooser.showDialog(menu.getBaseWin(), null);
     // second arg is optional approve button name override

     if (result != fileChooser.APPROVE_OPTION) {
       DebugEditIM.println(1, "Selection of a file aborted by the user.");
       return; // user canceled... (CLOSE_OPTION or CANCEL_OPTION)
     }
     String selFName = "";
     File selF = fileChooser.getSelectedFile();
     if (selF != null)
       selFName = selF.getAbsolutePath() + File.separator + selF.getName();
     if ((selF == null) || selF.isDirectory()) {
       result = fileChooser.CANCEL_OPTION;
       // refuse non-selections and directories
     } else if (verb.equals("saved")) {
       try {
         if ( (selF.getParent() != null) &&
              (!selF.getParentFile().canWrite()) ) {
           DebugEditIM.println(1, "Cannot write directory of:" + selFName);
           result = fileChooser.CANCEL_OPTION; // try again
         } else if (!selF.createNewFile()) {
           // atomic file creation: fails if file already existed
           // could also use simple exists() check here.
           //
           int confirmed = JOptionPane.showConfirmDialog(
             menu.getBaseWin(), "File exists, overwrite it?",
             "Input Method Editor Warning",
             JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
           if (confirmed != JOptionPane.YES_OPTION) {
             result = fileChooser.CANCEL_OPTION; // try again
           } else {
             DebugEditIM.println(1,
               "Warning: Will overwrite existing file: " + selFName);
           }
         } // confirmation overwrite
       } catch (IOException ioe) {
         DebugEditIM.println(0, "Ouch! File creation error?");
         ioe.printStackTrace();
         result = fileChooser.CANCEL_OPTION;
       } // try / catch
     } // save
     else { // merge or load
       if (!selF.canRead()) {
         DebugEditIM.println(1, "Cannot read file: " + selFName);
         result = fileChooser.CANCEL_OPTION; // try again
       }
     }
     if (result != fileChooser.APPROVE_OPTION)
       java.awt.Toolkit.getDefaultToolkit().beep();
   } while (result != fileChooser.APPROVE_OPTION);
   //
   int fileTypeCode = fileTypeComboBox.getSelectedIndex();
   fileTypeCode = (fileTypeCode < 0) ? 2 // none selected? then GIM!
     : typeCodes[fileTypeCode];          // translate type code
   String encoding = "ISO-8859-1";       // default is ISO (not "US-ASCII")
   if ((fileTypeCode == AssignObject.U8_FILE) ||
       (fileTypeCode == AssignObject.UNICODE_HUMAN) ||
       (fileTypeCode == AssignObject.YUDIT_FILE) // only comments need UTF-8
     )
     encoding = "UTF-8"; // other possibilities: UTF-16[LE,BE,]
   //
   String filename;
   try {
     filename = fileChooser.getSelectedFile().toURL().toString();
   } catch (java.net.MalformedURLException mue) {
     filename = "file://" + fileChooser.getSelectedFile().getName();
   } // catch
   // File ... = fileChoser.getCurrentDirectory(); ...
   //
   DebugEditIM.println(1, "title=" + title + ", command="
     + command + " => User selected file encoding="
     + encoding + ", type=" + fileTypeCode
     + " (" + fileTypeComboBox.getSelectedItem() + "), "
     + "name=" + filename );
   menu.tellListener(command + " " + encoding + " "
     + fileTypeCode + " " + filename);
 } // fileTypeRequest


 /* *** *** */


 /**
  * The main point where events are coming in.
  * <p>
  * this is an ActionListener, so it must handle actionPerformed. All
  * requests are coming in through this method. Each type of request
  * should be handled by calling one or more of the private methods
  * of this class. This is the common parser for all. Some of the
  * methods above throw events back by using menu.tellListener().
  * </p>
  * <p><pre>
  * Accepted commands:
  * loadlocale file - ask for a filename AND type to be loaded.
  * mergelocale file - as loadlocale file, but for merge.
  * savelocale file - ask for filename AND type for save.
  * </pre></p>
  */
 public void actionPerformed(ActionEvent e) {
   String command = e.getActionCommand();
   int modifiers = e.getModifiers();
   // Object source = e.getSource();
   java.util.StringTokenizer strtok =
     new java.util.StringTokenizer(command);
   try {
     String activity = strtok.nextToken();
     if ( activity.equals("loadlocale")  ||
          activity.equals("mergelocale") ||
          activity.equals("savelocale")  ) {
       String fileS = strtok.nextToken();
       if (!fileS.equals("file")) return; // other cases handled elsewhere
       if (activity.startsWith("load")) {
         DebugEditIM.println(2,
           "load locale file (name, type) popup requested!");
         // loadlocale file - ask for a filename AND type to be loaded
         fileTypeRequest("Load locale file for editing", "loadlocalefile");
           // returns savelocalefile filetype filename
       } // load
       else if (activity.startsWith("merge")) {
         DebugEditIM.println(2,
           "lmerge locale file (name, type) popup requested!");
         // mergelocale file - ask for a filename AND type to be merged
         // with the currently edited one ( *** may cause choice lists *** )
         // ( *** must use table2 if TWO keys map to the SAME string *** )
         fileTypeRequest(
           "Merge locale file with the loaded one for editing",
           "mergelocalefile");
           // returns savelocalefile filetype filename
       } // merge
       else { // save
         DebugEditIM.println(2,
           "save locale file (name, type, locale) popup requested!");
         // savelocale file - ask for a name AND type AND locale to be saved
         fileTypeRequest("Export to new locale file", "savelocalefile");
           // returns savelocalefile filetype filename locale
       } // savelocale
     } // loadlocale, savelocale, mergelocale
  } catch (Exception ex) {
    DebugEditIM.println(0, "Unparseable popup command: " + command);
    // ex.printStackTrace();
  }
 } // actionPerformed


} // class FileRequester

