/**
 * This class loads a font dynamically, either from file or from
 * a resource. Note that only the user of the font might be able to
 * handle certain glyphs in cut and paste, as other applications may
 * have no access to an appropriate font (as opposed to the situation
 * when using globally installed fonts).
 *  [ derived from package mpi.learn.fonts by Eric Auer 2003. ]
 *  [ Modified heavily to become a "smooth" system / dynamic  ]
 *  [ font interface for the Input Method Editor and others.  ]
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import java.awt.*;
import javax.swing.*;
import javax.swing.text.*; /* MutableAttributeSet */
import java.io.*;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.*;	/* Vector */

import guk.editIM.DebugEditIM;

/**
 * This class loads a font dynamically, either from file or from
 * a resource. Note that only the user of the font might be able to
 * handle certain glyphs in cut and paste, as other applications may
 * have no access to an appropriate font (as opposed to the situation
 * when using globally installed fonts).
 */
public class FontLoader {

    /**
     * private font object for the current INTERNAL font
     */
    static Font font = null;

    /**
     * internal font names
     */
    static Vector iFonts = new Vector(1 /* opt. initial size */);

    /**
     * debug level, as explained in DebugEditIM.
     */
    static int debug = 0;


    /**
     * The constructor preloads internal fonts and inits iFonts.
     * Sets debug level to default, which is quite un-verbose.
     */
    public FontLoader() {
      debug = 0;
      font = null;
      iFonts = new Vector(1); // growable, searchable, etc.
      DebugEditIM.println(3, "FontLoader()");
      loadFonts(); // loads font(s) and sets up iFonts
      DebugEditIM.println(1,
        "FontLoader() loaded " + iFonts.size() + " dynamic fonts");
    }


    /**
     * The constructor preloads internal fonts and inits iFonts.
     * Here we can pass the debug level.
     * @param debug Gives a debug
     * level from -1 to infinite (-1 or below mutes normal messages)
     */
    public FontLoader(int udebug) {
      debug = udebug;
      font = null;
      iFonts = new Vector(1); // growable, searchable, etc.
      DebugEditIM.println(3, "FontLoader()");
      loadFonts(); // loads font(s) and sets up iFonts
      DebugEditIM.println(1,
        "FontLoader() loaded " + iFonts.size() + " dynamic fonts");
    }



    /**
     * Find a font with the requested properties or derive it
     * from a known font.
     * @param nameF Name of the font family to be found or null
     * for default.
     * @param sizeF Size in points for the font. If 1 or smaller,
     * take size from oldFont or use default if no oldFont.
     * @param oldFont An already known font from which the new
     * font can be derived, or null.
     * @return A font which is as close as possible to the requested
     * one. Worst case is returning oldFont.
     */
    public Font fontSet(String nameF, int sizeF, Font oldFont) {
      int oldSize;
      String oldFamily;

      if (oldFont == null) {
        oldFont = new Font(null /* default */, 0 /* BOLD... */, 10);
        DebugEditIM.println(2, "fontSet uses default oldFont: " + oldFont);
      }
      oldSize = oldFont.getSize();
      oldFamily = oldFont.getFamily();

      DebugEditIM.println(3, "fontSet( nameF=" + nameF + ", sizeF=" + sizeF
        + ", oldFont=" + oldFont);

      if ((nameF == null) || nameF.equals("") || nameF.equals(oldFamily))
        nameF = ""; // value for "keep as is"
        // should detect unchanged dynamic font here, too.
      if ((sizeF < 1) || (sizeF == oldSize))
        sizeF = oldSize; // "keep as is"

      try {

        if (this.getInternalFonts().contains(nameF)) { /* internal dynamic font */

          Iterator iFontIter = iFonts.iterator();
          while (iFontIter.hasNext()) {
            font = (Font)(iFontIter.next());
            if (getInternalName(font).equals(nameF)) {
              DebugEditIM.println(2,
                "Match for nameF <" + nameF + ">: " + font);
              if (font.getSize() != sizeF)
                font = font.deriveFont((float)sizeF);
              return font;
            } else {
              DebugEditIM.println(4, "Not matching: <"
                + getInternalName(font) + "> and <" + nameF + ">");
            } // no match
          } // while
          DebugEditIM.println(1, "Bummer! Thought I would know a font, "
            + "but cannot find it anymore: \n<"
            + nameF + "> " + sizeF + "pt, dynamical");
          return oldFont.deriveFont((float)sizeF);

        } /* "* internal *" font loading and reactivating */
        else
        { /* System fonts, not dynamic fonts */

          Font aFont = oldFont;
          if (!aFont.getFamily().equals(nameF)) { // family change
            DebugEditIM.println(2,
              "System font search for font of family: " + nameF);
            Font [] allFonts = GraphicsEnvironment.
              getLocalGraphicsEnvironment().
              getAllFonts();
            // stupid: I cannot chose a font BY FAMILY apart
            // from using attribute editing in swing components???
            for (int i = 0; i < allFonts.length; i++) {
              if (allFonts[i].getFamily().equals(nameF)) {
                DebugEditIM.println(2,
                  "System font (family match): " + allFonts[i]);
                aFont = allFonts[i].deriveFont((float)sizeF);
                // could check attributes here (plain, bold, ...)
                return aFont; // just return first match!
              }
            } // for

            // if the name was not of a family, it might be a font name

            aFont = new Font(nameF, Font.PLAIN, sizeF);
            DebugEditIM.println(2,
              "System font (trying name <" + nameF + ">): " + aFont);
            return aFont;
          } // family change

          // same family, only change size, if needed:
          if (aFont.getSize() != sizeF) {
            DebugEditIM.println(2, "System font resize: "
              + aFont.getSize() + "pt to " + sizeF + "pt");
            return aFont.deriveFont((float)sizeF);
          }

          // nothing changed at all
          DebugEditIM.println(2, "Fonts unchanged");
          return aFont;

        } // EOF internal fonts

      } catch (Exception exc) {
        DebugEditIM.println(1, "Bummer! Could not fontSet to <"
          + nameF + "> " + sizeF + "pt, old Font is " + oldFont);
        exc.printStackTrace();
        return oldFont;
      }

    } // EOF fontSet


    /**
     * Tell which fonts have been loaded dynamically.
     * @return A list (Vector with Strings) of the internal names
     * of the dynamically loaded fonts.
     */
    public Vector getInternalFonts() {
      Vector iFontNames = new Vector();
      Iterator iFontIter = iFonts.iterator();
      String aName;
      while (iFontIter.hasNext()) {
        aName = getInternalName( (Font)(iFontIter.next()) );
        DebugEditIM.println(4, "Dynamic font internal name: " + aName);
        iFontNames.add(aName);
      } // while
      return iFontNames;
    }


    /**
     * Get the special &quot;internal&quot; name of a font.
     * Just prefixes the font family name with
     * &quot;Dynamic:&nbsp;&quot; in order to do this.
     * Generates the internal String name for the given aFont,
     * for comparison. Of course, I could encapsulate the loaded fonts
     * in a class with an own equals() or an own conversion to string...
     * I want a certain amount of "do not care" properties, so the normal
     * to string or equals() is not appropriate.
     * @param aFont An arbitrary font, but should be a dynamic one.
     * @return A special name for that font as described above.
     */
    public String getInternalName(Font afont) {
      return "Dynamic: " + afont.getFamily();
    }



    /**
     * Print the name of the font with verbosity level 0.
     * @param thisFont is "named" (size, family, name) to STDOUT
     */
    private void fontName(Font thisFont) {
      if (thisFont == null) return;
        DebugEditIM.println(0, "<"
          + thisFont.getName() + "> of family <"
          + thisFont.getFamily() + "> ("
          + thisFont.getSize() + "pt)");
    } // EOF fontName


    /**
     * Initializes the iFonts font collection.
     * Load dynamic fonts and remember them in iFonts.
     */
    public /* better private? */ void loadFonts() {
      InputStream fontS;
      if (font != null) {
        // have loaded font before :-)
        return;
      }
      try {
        fontS = new FileInputStream("arialuni.ttf");
        font = Font.createFont(Font.TRUETYPE_FONT, fontS);
        iFonts.add(font);
        DebugEditIM.println(0, "Using arialuni.ttf font");
      } catch (Exception ex) {
        DebugEditIM.println(1, "./arialuni.ttf cannot be read, "
          + "trying ./cyberbit.ttf now.");
          font = null;
      }
      try {
        fontS = new FileInputStream("cyberbit.ttf");
        font = Font.createFont(Font.TRUETYPE_FONT, fontS);
        iFonts.add(font);
        DebugEditIM.println(0, "Using cyberbit.ttf font");
      } catch (Exception ex) {
        DebugEditIM.println(1, "./cyberbit.ttf cannot be read, "
          + "trying resource [data/Cyberbas.ttf] next.");
        font = null;
      }
      try {
        fontS =	this.getClass().getResourceAsStream(
          "data/Cyberbas.ttf");
        font = Font.createFont(Font.TRUETYPE_FONT, fontS);
        iFonts.add(font);
        DebugEditIM.println(0, "Using [Cyberbas.ttf] font resource");
          // path relative to .class file path!?
          // getResourceAsStream returns an InputStream, works as
          // getResource().openStream() but maps errors to returning null.
      } catch (Exception exc) {
        DebugEditIM.println(1,
          "Could not find / load Cyberbas.ttf font resource");
        font = null;
      }
    } // EOF loadFonts


} // FontLoader

