/**
 * The graphical user interface helpers for the GlyphTable.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;

// import guk.editIM.MenuHelpers;
// import guk.editIM.GTVisibility;
// import guk.editIM.DebugEditIM;
import javax.swing.table.*; // table models and related
import javax.swing.*; // JTable and similar
import java.awt.*; // color stuff, renderer stuff
import java.util.Hashtable;


/**
 * The graphical user interface helpers for the GlyphTable.
 * Implements a scroll helper and something that is almost
 * a TableCellRenderer. Used by {@link GlyphTable GlyphTable}.
 */
public class GTRenderer {


  /**
   * For efficiency, we keep an handle to the main data
   * structure of the GlyphTable for which we are rendering.
   */
  Hashtable usedSlots = null;


  /**
   * The visor objects keeps track of which internal rows
   * are externally visible and how row numbers are mapped.
   * Can be controlled by the processCommand() inside.
   */
  GTVisibility visor = null;


  /**
   *  The precomputed color for each glyph. (Could probably save
   *  some RAM here by using other data types!?). I assume that
   *  figuring out which Unicode character range a glyph has
   *  would be expensive, so the color is stored for each glyph!
   */
  Color [] fgColor = new Color [65536];
  /**
   * similar as fgColor, but for the background color.
   * more important, as the fgColor is very dark and looks
   * almost like black.
   */
  Color [] bgColor = new Color [65536];


  /**
   * some abstraction-breaking variables that help us to remote-
   * control the table in which we are loaded. initialized by
   * setScroller below.
   */
  JTable myTable = null;
  /**
   * knowing our own viewport allows us to scroll the table for
   * which we are the data model.
   */
  JViewport myViewport = null;
  /**
   *
   */
  MenuHelpers menu = null;


  /**
   * The constructor initializes the colors.
   * For each internal row, the Unicode range of the glyph
   * in it is determined. For each Unicode range, a color
   * is determined on pseudo-random basis. Background colors
   * are very light, foreground colors are almost black.
   * @param contents The hashtable from which we can read the
   * contents of the table in native format, for highlighting.
   * @param visitor An instance of GTVisibility, to know the
   * mapping of visible to internal rows.
   * @param menuHelper A MenuHelpers instance for font and
   * color calculations.
   */
  public GTRenderer(Hashtable contents, GTVisibility visitor) {
    usedSlots = contents;
    visor = visitor;
    menu = new MenuHelpers(null);
      // used: toHex(), hueByHash(), fontSync()
    int i, hash, old;
    float diff;
    System.out.print("Setting up GlyphTable colors...");
    // we use hardcoded 0..65535 ranges instead of Character
    // .MIN_VALUE and .MAX_VALUE here (would not really save
    // time, and with 32bit Unicode, it would waste time)
    old = -1;
    for (i = 0; i < 65536; i++) {
      java.lang.Character.UnicodeBlock cub =
        java.lang.Character.UnicodeBlock.of((char)i);
      hash = (cub == null) ? -1 : cub.hashCode();
      if (old == hash) { // save time, color unchanged
        fgColor[i] = fgColor[i-1];
        bgColor[i] = bgColor[i-1];
      } else { // figure out new color
        diff = (1.0f / 32) * (i >> 7);
          // visible change by 1/32 color circle every 128 glyphs...
        fgColor[i] = menu.hueByHash(Color.black, hash , 0.0f + diff);
        bgColor[i] = menu.hueByHash(Color.white, hash , 0.5f + diff);
        old = hash; // remember that we already did this
      }
    } // for
    System.out.println(" done.");
  } // constructor


  /**
   *  Know which table we belong to and which viewport, so that
   *  we can remote control them.
   *  Gather the needed information (who are our viewport and table
   *  parents) to allow us to scroll ourselves, kind of breaking the
   *  abstraction barrier between table data model and table
   * @param vp The viewport in which we can be seen, used for program
   * controlled scrolling.
   * @param tab The table to which we belong to (to know the layout,
   * for scrolling).
   * @param theMenu Can be set if you want a specific font setting.
   */
  public void setScroller(JViewport vp, JTable tab,
    MenuHelpers theMenu) {
    myViewport = vp;
    myTable = tab;
    if (theMenu != null) menu = theMenu;
  } // setScroller


  /**
   *  Scroll the viewport to a certain glyph / internal row.
   *  Helper method that translates internal to visible row numbers
   *  and scrolls there (combined method: nothing else yet needs that
   *  translation). Yet another public method which MapTables do not
   *  have. When the requested glyph is not visible, the scrolling
   *  is to the next visible glyph in in the area.
   *  @param row The internal row number (glyph) to which we should
   *  scroll. You have to set a viewport or nothing will scroll.
   */
  public void scrollTo(int row) {
    int row2;
    row2 = visor.internalToRow(row);
    if (row2 < 0) {
      DebugEditIM.println(0,
        "Tried to scroll to invisible glyph " + row);
      return;
    }
    DebugEditIM.println(2, "Scroll to row " + row2 + " glyph " + row);
    row = row2;
    if ((myTable != null) && (myViewport != null)) {
      java.awt.Rectangle rect = myTable.getCellRect(row, 0, true);
      // get dimensions of cell(row,0), include_margins is true.
      // myViewport.scrollRectToVisible(rect) only makes cell VISIBLE,
      // but we want to scroll so that the TOP LEFT Point cell is on top:
      myViewport.setViewPosition(rect.getLocation());
      DebugEditIM.println(3, "setViewPosition to " + rect.getLocation());
    } else {
      DebugEditIM.println(2, "no viewport to scroll");
      // not fatal, not even the message is really important.
    }
  }


  /**
   * The renderer: Take old renderer and improve the ToolTip
   * and maybe other things before passing on the result.
   * This table model provides its own renderer, which sets the
   * color according to which unicode range holds the affected row.
   * Also adds some useful information about the glyph to all fields
   * that happen to contain only one glyph. The information will end
   * up in the ToolTip. This is a SPECIAL method but very similar to
   * getTableCellRendererComponent(...) in AbstractTableModel. The
   * difference is the additional renderer argument here.
   * <p><i>
   * WARNING: TableModels normally do not implement TableCellRenderer
   * and this exception (activated via table.setDefaultRenderer(...))
   * means that we have to be CAREFUL with coordinates: JTable and
   * renderer stuff uses VIEW coordinates, but the Model uses MODEL
   * coordinates. The user may have swapped (by dragging) the columns!
   * </i></p>
   * <p><b>(Unicode needs HTML 4.0 to render, JLabels - the
   * the used TableCellRenderers are derived from them - only render
   * ISO-8859-1 HTML 3.2, so we see boxes for nonprintable chars
   * when HTML highlighting is active).
   * </b></p>
   * @param renderer A basis for our calculations. The real
   * getTableCellRendererComponent method does not have this extra.
   * @see javax.swing.table.TableCellRenderer
   */
  public Component getTableCellRendererComponent(
    JTable table, Object value, boolean isSelected,
    boolean hasFocus, int row, int column, Component renderer) {
    //   table can be null, row can be -1 for the header.
    //   configures the renderer to draw the described cell
    //
    // If we were something that extends AbstractTableModel, we could:
    // Component renderer = super.getTableCellRendererComponent(
    //   table, value, isSelected, hasFocus, row, column);
    int i = visor.rowToInternal(row);
      // take invisible rows into account
    int modelColumn = table.convertColumnIndexToModel( column );
    Integer rowObj = new Integer(i); // keys have to be Objects
      // note that we use the internal row number here!
    String textual = "" + value;
    //
    if ((i >= 0) && (i < 65536)) {
      renderer.setForeground(fgColor[i]);
      renderer.setBackground(bgColor[i]);
    }
    menu.syncFont(renderer);
    //
    if ( (modelColumn == 1) && usedSlots.containsKey(rowObj) ) {
      AssignObject ao = (AssignObject)(usedSlots.get(rowObj));
      String htmlText = ao.exportKeys(AssignObject.UNICODE_HTML);
        // from Java 1.3 on, JLabels and other JStuff accepts HTML!
        // we take our chance and HTMLize any used key sequence :-).
        // (textual stays unchanged!)
      if (htmlText.indexOf("&#") > 0) {
        DebugEditIM.println(4, "not HTMLizing: Unicode in use!");
        /** Actually, this simple pattern matching even triggers
        *   for ISO-8859-1 Unicode, so we are sometimes turning
        *   off the JLabel HTML 3.2 too early. */
      } else if (renderer instanceof JLabel) {
        DebugEditIM.println(5, "text: " + textual);
        ((JLabel)renderer).setText(htmlText);
        // bad luck otherwise, cannot pass on new, HTMLized, value.
        DebugEditIM.println(5, "HTMLized: " + htmlText);
      } else {
        DebugEditIM.println(3, "No HTMLizeable renderer for: " + textual);
      }
    } // Keys that we know about
      // (special treatment for glyphs is below: single-key key
      // sequences are treated like glyphs there, bonus ToolTip)
    //
    if (textual.length() == 1) { // show unicode info if ONE char or key
      // assume (mostly) useful settings, thanks to the superclass method
      char ch = ((String)value).charAt(0);
      if ( (modelColumn == 0) && (renderer instanceof JLabel) ) {
        ((JLabel)renderer).setHorizontalTextPosition(JLabel.CENTER);
        ((JLabel)renderer).setHorizontalAlignment(JLabel.CENTER);
      }
      if (renderer instanceof JComponent) {
        ((JComponent)renderer).setToolTipText("\\u" + menu.toHex((int)ch)
          + " [" + ch + "] in Range " + Character.UnicodeBlock.of( ch ));
      }
      // we can NOT control the tool tip font, so most glyphs remain
      // undisplayable in the "[x]" substring. We would have to extend the
      // JToolTip class to create a new renderer. Or is setFont enough?
      /** @todo: modify our JToolTip renderer to use an Unicode font? */
    } else { // textual length not 1
      // hoping for some useful ToolTip from the superclass method...
    } // ToolTip if/then/else thing
    //
    return renderer;
    //
  } // getTableCellRendererComponent


  /**
   * Process command strings.
   * <p><pre>
   * process a command sent by e.g. the GUI menu system:
   *
   * jumpto 1234 (scrolls to that glyph).
   * jumpfield CHAR (scrolls to that glyph - char may be escaped).
   * </pre></p>
   * @param modifiers Modifiers use the normal ActionEvent bit masks.
   * @param checked Some commands use this as additional boolean input.
   */
  public void processCommand(String command, int modifiers, boolean checked) {
    // modifiers are 0 and 16 for keyboard and Lmouse, but other bits
    // can be set for shift, ctrl, alt, Mmouse, Rmouse. Not yet used.
    String action, pos1s, pos2s;
    int pos1 = 0;
    int pos2 = 0;
    // if MapTable itself already parses commands:
    // super.processCommand(command, modifiers, checked);
    try {
      java.util.StringTokenizer strTok =
        new java.util.StringTokenizer(command);
      // default tokenizer, using " \r\n\t" as delimiters
      if (!strTok.hasMoreTokens()) return; // need command
      action = strTok.nextToken();
      if (!strTok.hasMoreTokens()) return; // need at least one argument
      pos1s = strTok.nextToken(" \t\n\r\f-"); // get first argument
      // second argument can be separated by "-" or space
      if (strTok.hasMoreTokens()) {
        pos2s = strTok.nextToken();
      } else {
        pos2s = "";
      }
      // commands with single argument:
      if (action.equals("jumpto")) {
        //
        scrollTo(Integer.parseInt(pos1s, 10)); // parse as decimal
        //
      } // jumpto
      else if (action.equals("jumpfield")) {
        //
        if (pos1s.length() == 1) { // jump to glyph entered by user
          scrollTo( (int)(pos1s.charAt(0)) );
        } else
        if ( pos1s.startsWith("\\u") || // unicode char in escaped form
            pos1s.startsWith("0x") ) {  // hex number with proper prefix
          scrollTo( Integer.parseInt(pos1s.substring(2), 16) );
        } else { // assume dec number, if it fails, cancel the jump by catch
          scrollTo( Integer.parseInt(pos1s.trim(), 10) );
        }
        //
      } // jumpfield
      else {
        //
        return; // ignore all commands that are not for us!
        //
      } // ignored commands
      //
      DebugEditIM.println(1, action + ", " + pos1s + ", " + pos2s + ", "
        + (checked ? "" : "un") + "checked, modifiers=" + modifiers);
    } catch (Exception exc) {
      DebugEditIM.println(0, "error in command: " + command
        + " [modifiers=" + modifiers + ", checked=" + checked + "]");
      if (!command.startsWith("jumpfield")) // if not user fault...
        exc.printStackTrace();
      // no fatal error, just ignore the malformed command and continue
    } // catch
  } // processCommand


} // public class GTRenderer implements TableCellRenderer

