/**
 * Visibility and internal versus user row mapping for the
 * glyph table, including the required command processing.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;

import javax.swing.table.*; // table models and related
import java.util.Hashtable;


/**
 * Visibility and internal versus user row mapping for the
 * glyph table, including the required command processing.
 */
public class GTVisibility {


  /**
   * If this is true, mapped glyphs will always be visible
   * when the hideUnmapped state is on. Otherwise, they can
   * be hidden even in that state by showall / showpart settings.
   */
  static protected final boolean AUTOMAPPED = true;

  /**
   *  The current assignment between visible rows and internal rows,
   *  some rows might be invisible so the values might differ!
   *  used at different places, maintained by processCommand.
   *  speed does matter here, so we have a lookup array.
   *  <pre>
   *  rowToGlyph: know glyph from user row (for table interface)
   *  glyphToRow: know user row from glyph (for scrolling)
   *  </pre>
   */
  int rowToGlyph [] = new int [65536]; // visible row number to internal
  /**
   * like rowToGlyph but for translating internal (glyph) to visible
   * row number.
   */
  int glyphToRow [] = new int [65536]; // internal to visible row number


  /**
   *  Describes the visibility status of all lines.
   *  <pre>
   *  showAll: which glyphs are certainly visible?
   *  showPart: which glpyhs are specially visible?
   *    the idea is that you can toggle showAll without having
   *    to update showPart. If either is true, the row is visible.
   *  </pre>
   */
  boolean showAll [] = new boolean [65536]; // showall overrides showpart
  /**
   * ORed together with showAll to determine visibility.
   */
  boolean showPart [] = new boolean [65536]; // see above
  /**
   * current size of the table, number of visible rows.
   */
  int visibleRows = 0; // tells about rowToGlyph limits
  /**
   * whether to hide rows where no key sequence is assigned.
   * Useful to get an overview of the current mapping by hiding
   * all unmapped glyphs.
   * (protected to have it javadoc-ed even without -private flag ;-)).
   */
  protected boolean hideUnmapped = false;


  /**
   * The contents of the table, so that we can stay up to date
   * about which internal rows are in use, for the hideUnmapped
   * feature.
   */
  java.util.Map rowContents = null;


  /**
   * The constructor initializes the arrays. It also receives a
   * map with contents, to be able to know which rows are used
   * (for the hideUnmapped feature, see processCommand()).
   * @param contents A map where an Integer key retrieves the
   * contents for the internal row selected by the Integer value.
   * The map will be queried from time to time.
   */
  public GTVisibility(java.util.Map contents) {
    int i;
    rowContents = contents;
    visibleRows = 0; // no rows at all visible!
    for (i = 0; i < 65536; i++) {
      glyphToRow[i] = -1; // all glpyhs invisible
      rowToGlyph[i] = -1; // no row produces a glyph
      showAll[i] = false; // not generally visible
      showPart[i] = false; // not specially visible
    }
    processCommand("showall 32-127", 0, true); // enable something
  } // constructor


  /**
   * Figure out how many rows are currently to be shown.
   * @return The number of rows that are currently visible.
   */
  public int getVisibleRowCount() {
    return visibleRows; // must be efficient!
  }


  /**
   * Ask if the hideUnmapped feature is on.
   * @return Whether unused rows are hidden (even if showall
   * or showpart would enable their visibility).
   */
  public boolean isHideUnmappedOn() {
    return hideUnmapped;
  }


  /**
   *  Translate visible row to glyph number / internal row.
   *  Helper method to translate from visible to internal row
   *  numbers - some rows might be configured as invisible.
   *  @param row The number of a visible row (0 is the topmost
   *  visible row).
   *  @return The number of an internal row. If all rows were
   *  visible, input and output of this function were equal.
   */
  public int rowToInternal(int row) {
    if (row > (visibleRows-1)) return -42;
    if (row < 0) return row;
    return rowToGlyph[row];
  } // rowToInternal


  /**
   *  Translate glyph number / internal row to visible row.
   *  @param row The number of a glyph / internal row.
   *  @return The number of a visible row, possibly approximated,
   *  where the requested internal row is visible.
   */
  public int internalToRow(int row) {
    if (row > 65535) return -42;
    if (row < 0) return row;
    return glyphToRow[row];
  } // internalToRow


  /**
   *  Determine which internal rows are visible, based on
   *  the internal showAll and showPart arrays and on hideUnmapped.
   *  You can modify those arrays and hideUnmapped by using the
   *  processCommand() interface.
   *  Helper method that updates rowToGlyph and glyphToRow and
   *  the looks of the table. not performance critical. uses the
   *  showAll and showPart boolean arrays as source.
   */
  public void updateVisibility() {
    int i = 0; // current internal row aka. glyph number
    int j = 0; // current visible row
    visibleRows = 0; // rebuild from assuming 0 rows visible
    for (j = 0; j < 65536; j++) glyphToRow[j] = -1; // flush list
    j = 0; // visible row counter
    for (i = 0; i < 65536; i++) {
      if (hideUnmapped) {
        Integer rowObj = new Integer(i); // keys must be Objects
        if ( (rowContents == null) ||    // sic!
             (rowContents.containsKey(rowObj)) ) {
          // empty rows are only hidden if we know the rowContents.
          if (showPart[i] || showAll[i] ||
            AUTOMAPPED) { // visible glyph row
            rowToGlyph[j] = i;
            j++;
          } // enabled
        } // not empty, actually contains an AssignObject
      } // hide empty
      else { // visibility only depends on showPart and showAll
        if (showPart[i] || showAll[i]) { // visible glyph row
          rowToGlyph[j] = i;
          j++;
        } // enabled
      } // show all enabled
    } // for glyphs
    if (j == 0) { // avoid empty table
      rowToGlyph[0] = 32; // map space
      glyphToRow[32] = 0; // map space
      j++;
    } // avoid empty table
    visibleRows = j; // remember the number of visible glyphs found
    for ( ; j < 65536; j++) rowToGlyph[j] = -1; // fill rest with "none"
    //
    for (j = 0; j < 65536; j++) {
      if (rowToGlyph[j] >= 0) // we found a visible glyph
        glyphToRow[rowToGlyph[j]] = j; // remember where it can be seen
    } // for visibles
    int lastseen = 0;
    for (j = 0; j < 65536; j++) {
      if (glyphToRow[j] >= 0) lastseen = glyphToRow[j];
      glyphToRow[j] = lastseen; // fill gaps with "approximate" values
    } // for visibles
  } // updateVisibility


  /**
   * Process command strings.
   * <p><pre>
   * process a command sent by e.g. the GUI menu system:
   *
   * showpart 12-34 (show that glyph range, using PART bitmap),
   * showall 12-34 (show that glyph range, using ALL bitmap),
   * hideunmapped glyphs (hide all unmapped glyphs from the table)
   * (hideunmapped, showall and showpart also use the checked argument)
   * </pre></p>
   * @param modifiers Modifiers use the normal ActionEvent bit masks.
   * @param checked Some commands use this as additional boolean input.
   */
  public void processCommand(String command, int modifiers, boolean checked) {
    // modifiers are 0 and 16 for keyboard and Lmouse, but other bits
    // can be set for shift, ctrl, alt, Mmouse, Rmouse. Not yet used.
    String action, pos1s, pos2s;
    int pos1 = 0;
    int pos2 = 0;
    // if MapTable itself already parses commands:
    // super.processCommand(command, modifiers, checked);
    try {
      java.util.StringTokenizer strTok =
        new java.util.StringTokenizer(command);
      // default tokenizer, using " \r\n\t" as delimiters
      if (!strTok.hasMoreTokens()) return; // need command
      action = strTok.nextToken();
      if (!strTok.hasMoreTokens()) return; // need at least one argument
      pos1s = strTok.nextToken(" \t\n\r\f-"); // get first argument
      // second argument can be separated by "-" or space
      if (strTok.hasMoreTokens()) {
        pos2s = strTok.nextToken();
      } else {
        pos2s = "";
      }
      // commands with single argument:
      if (action.equals("hideunmapped")) {
        //
        hideUnmapped = checked; // hide all rows with empty strings?
        this.updateVisibility(); // recalculate list of visible rows
        //
      } // hideunmapped
      else if (action.equals("showall") || action.equals("showpart")) {
        //
        // find numerical arguments:
        pos2 = Integer.parseInt(pos2s, 10); // NOW we dare to parse as int
        pos1 = Integer.parseInt(pos1s, 10);
          // or use Integer.valueOf(string, radix).intValue()
        if ((pos1 < 0) || (pos1 > 65535) || (pos2 < 0) || (pos2 > 65535))
          return; // numeric arguments not suitable for following commands
        // commands with numerical arguments:
        if (action.equals("showall")) {
          for (int i = pos1; i <= pos2; i++)
            showAll[i] = checked; // update array region
          this.updateVisibility(); // recalculate mapping and repaint
        } // showall
        else if (action.equals("showpart")) {
          for (int i = pos1; i <= pos2; i++)
            showPart[i] = checked; // update array region
          this.updateVisibility(); // recalculate mapping and repaint
        } // showpart
        //
      } // showall or showpart
      else {
        //
        return; // ignore all commands that are not for us!
        //
      } // ignored commands
      //
      DebugEditIM.println(1, action + ", " + pos1s + ", " + pos2s + ", "
        + (checked ? "" : "un") + "checked, modifiers=" + modifiers);
    } catch (Exception exc) {
      DebugEditIM.println(0, "error in command: " + command
        + " [modifiers=" + modifiers + ", checked=" + checked + "]");
      exc.printStackTrace();
      // no fatal error, just ignore the malformed command and continue
    } // catch
  } // processCommand


} // public class GTVisibility

