/**
 * An implementation of the glyph selection and
 * edit keymap for glyph table in terms of a MapTable
 * (which extends AbstractTableModel).
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */



package guk.editIM;

import guk.editIM.MapTable;
import guk.editIM.GTVisibility;
import guk.editIM.GTRenderer;
import javax.swing.table.*; // table models and related
import javax.swing.*; // JTable and similar
import java.awt.*; // color stuff, renderer stuff
import java.util.Hashtable;


/**
 * A class to represent a table of all unicode glyphs,
 * where the user can control which lines are visible.
 * Only for glyphs where a key sequence exists,
 * an {@link AssignObject AssignObject} is stored.
 * The lines for
 * un-assigned glyphs are not stored but calculated.
 */
public class GlyphTable extends MapTable
  implements TableCellRenderer {


  /**
   *  the glyph value is simply (char) for any internal
   *  row i, but externally, not every row is visible
   *  all the time... When key sequences get assigned to
   *  glyphs, we use AssignObjects. Because an array for
   *  those would be sparse, we use an Hashtable.
   */
  Hashtable usedSlots = new Hashtable();


  /**
   * For some geometry manipulations, we can make use of
   * an object that holds our host JTable (or null).
   */
  JTable myTable = null;



  /**
   * The visor objects keeps track of which internal rows
   * are externally visible and how row numbers are mapped.
   * Can be controlled by the processCommand() inside.
   */
  GTVisibility visor = new GTVisibility(usedSlots);


  /**
   * The renderman object provides us with a TableCellRenderer
   * implementation and some scrollTo(row) method, which
   * can be controlled by the processCommand() inside.
   */
  GTRenderer renderman = new GTRenderer(usedSlots, visor);


  /**
   * The constructor is quite simple now, as the
   * {@link GTVisibility GTVisibility} and
   * {@link GTRenderer GTRenderer} classes are doing the
   * more specialized tasks. Those also cause some number
   * crunching when their objects are instantiated.
   */
  public GlyphTable() {
    this.processCommand("showall 32-127", 0, true); // enable something
  } // constructor


  /**
   *  Know which table we belong to and which viewport, so that
   *  we can remote control them. Just passed on to the renderman.
   *  Obviously breaks some abstraction barriers.
   *  @see GTRenderer
   */
  public void setScroller(JViewport vp, JTable tab, MenuHelpers menu) {
    renderman.setScroller(vp, tab, menu);
    myTable = tab;
  } // setScroller


  /* *** *** */


  /**
   *  Overridden because we use a Hashtable, no Vector.
   *  removes all mappings, Does update visibility.
   *  @return returns false on failure.
   */
  public boolean flushTable() {
    usedSlots.keySet().clear();
      // the Set supports clear, but the Map does not!?
      // cannot use usedSlots = new ..., because visor
      // would get out of sync. Solution would be to let
      // visor receive updated Maps through some method.
    if (visor.isHideUnmappedOn()) visor.updateVisibility();
      // only the contents, not the showall and showpart
      // settings did change.
    changedFlag = false;
    return true;
  } // flushTable


  /**
   *  Overridden because we use a Hashtable, no Vector.
   *  @return returns the Collection with all entries
   *  (for saving them).
   */
  public java.util.Collection readTable() {
    return usedSlots.values();
      // (Collection elements are identical to the ones in the
      // Hashtable, can be modified and removed, but not added.)
  } // readTable


  /**
   *  Overridden because we use a Hashtable, no Vector.
   *  merges an entry into the table,
   *  Does not update visibility.
   *  @return returns false on failure.
   */
  public boolean addEntry(AssignObject ao) {
    String glyphs = ao.getGlyphs();
    Integer keyObj = null;
    if (glyphs.length() != 1)
      return false; // cannot fit in here
    keyObj = new Integer( (int)(glyphs.charAt(0)) );
      // keys must be Objects
    if (usedSlots.containsKey(keyObj))
      return false; // only one A.O. per glyph possible in here
    Object oldAO = usedSlots.put(keyObj, ao);
      // oldAO will be null here
    // note that we do not set the change flag here
    return true; // success
  } // addEntry


  /**
   *  Overridden because we use a Hashtable, no Vector.
   *  removes an entry from the table,
   *  Does not update visibility.
   *  @return returns false on failure, for example when
   *  trying to remove a nonexisting entry.
   */
  public boolean removeEntry(AssignObject ao) {
    if (ao == null) return false;
    String glyphs = ao.getGlyphs();
    if (glyphs.length() != 1)
      return false; // cannot be from here
    Object oldAO = usedSlots.remove(
      new Integer((int)(glyphs.charAt(0))) );
    return (oldAO != null);
      // if oldAO is null, we tried to remove a nonexisting element
    // note that we do not set the change flag here
  } // removeEntry


  /* *** *** */


  /**
   *  Overridden as it differs from normal MapTable. All things
   *  that are already as wanted in a normal MapTable are not
   *  overridden in GlyphTable, e.g. getColumnCount()...
   *  The row count depends on which rows are visible. The
   *  number of internal rows is always 65536.
   */
  public int getRowCount() {
    return visor.getVisibleRowCount(); // must be efficient!
  }


  /**
   *  Finds and returns an appropriate keys / glyphs assignment,
   *  or to be more exact, the requested field of it.
   *  @return Returns a calculated value with empty key sequence
   *  and only the (char)row glyph for unused rows, and
   *  the real keys, glyphs or comments for used rows.
   */
  public Object getValueAt(int row, int col) {
    row = visor.rowToInternal(row);
      // translate visible into internal rows
    if (row < 0) {
      DebugEditIM.println(0, "getValueAt(" + row + ", " + col + ") ??");
      // return null;
      row = 32; // show the data of the glyph ' '
    }
    if (col == 0) return "" + (char)row; // dynamically generated :-)
      // by definition, even when we have something in the hashtable,
      // the glyph string must consist only of (char)row, because the
      // user cannot edit it.
    if (col >= 1) {
      Integer rowObj = new Integer(row); // keys must be Objects
      AssignObject ao = (AssignObject)(usedSlots.get(rowObj));
      if (ao == null)
        return ""; // dynamically empty :-)
      if (col == 1) {
        return ao.exportKeys(transferFormat);
        // for col 0, we would use exportGlyphs.
      } else if (col == 2) {
        return ao.getComments();
      }
    };
    return null;
  }


  /* *** *** */


  /**
   *  Overridden for dynamic title depending on
   *  the visor hideUnmapped status.
   *  Overridden to give the user some hint about the nature
   *  of this kind of table and how it differs from a normal
   *  MapTable...
   */
  public String getColumnName(int col) {
    if (col == 0) return "Glyphs";
    if (col == 1) {
      if (visor.isHideUnmappedOn()) {
        return "(Used) Key Sequences";
      } else {
        return "Key Sequences";
      }
    }
    if (col == 2) return "Comments";
    return "???";
  } // getColumnName



  /**
   *  Overridden to disallow editing of the glyph column.
   *  The glyphs are generated, not stored, so you cannot
   *  edit them.
   */
  public boolean isCellEditable(int row, int col) {
    // ignores row, all rows are treated the same here
    if (col == 0) return false;
      // in this UPPER table, glyphs are NOT editable
      // how can we COPY and PASTE the readonly glyph anyways???
      /** We have the glyph button and palette GUI for that... */
    if (col == 1) return true; // assigned sequences are editable
    if (col == 2) return true; // comments are editable
    return false;
  }


  /**
   *  Update the stored Hashtable of AssignObjects.
   *  Overridden: modifies or creates AssignObject in the Hashtable
   *  rather than using a sparse array of Strings. AssignObjects
   *  are removed when key sequences go empty, which may cause
   *  an update of the visibility status.
   */
  public void setValueAt(Object val, int row, int col) {
    /** *** would update running GUK IM FSM here *** */
    DebugEditIM.println(2,
      "setValueAt: [x=" + col + ", y=" + row + "] to: " + val);
    row = visor.rowToInternal(row);
      // translate visible into internal rows
    if (col == 0) {
      DebugEditIM.println(2, "Editing ignored: row=" + row + ", column="
          + col + ", old=" + ((char)row) + ", new=" + val.toString());
      return; // do not reach the point where the value is updated
    }
    if ( (col == 1) &&
         ( (val == null) || (((String)val).length() == 0) ) ) {
      Integer rowObj = new Integer(row); // keys must be Objects
      AssignObject oldAO = (AssignObject)(usedSlots.remove(rowObj));
      if (visor.isHideUnmappedOn()) visor.updateVisibility();
        // yet another line to be hidden, so update the mapping.
      return; // simply remove entries for which no keys are assigned
      // (even if COMMENTS are assigned to them!)
    }
    if (col >= 1) {
      Integer rowObj = new Integer(row); // keys must be Objects
      AssignObject ao = (AssignObject)(usedSlots.get(rowObj));
      if (ao == null) {
        ao = new AssignObject();
        ao.setGlyphs("" + ((char)row)); // fixed value
      }
      if (col == 1) {
        ao.importKeys((String)val, transferFormat);
      } else if (col == 2) {
        ao.setComments((String)val);
      }
      AssignObject oldAO = (AssignObject)(usedSlots.put(rowObj, ao));
        // could now export keys from both AO, to check for differences
      this.fireTableCellUpdated(row, col); // repaint etc.
      changedFlag = true; // could check for ACTUAL change here
        // (check if we had no AssignObject of same content already)
    }
    /* throw exception otherwise, or just ignore... */
  }


  /* *** *** */


  /**
   * The renderer: Use superclass method and improve.
   * This is a very nonstandard place for this method.
   * Most of the work is done by the renderman object.
   * @see GTRenderer
   */
  public Component getTableCellRendererComponent(
    JTable table, Object value, boolean isSelected,
    boolean hasFocus, int row, int column) {
    // table can be null, row can be -1 for the header.
    // configures the renderer to draw the described cell
    Component renderer = super.getTableCellRendererComponent(
      table, value, isSelected, hasFocus, row, column);
    renderer = renderman.getTableCellRendererComponent(
      table, value, isSelected, hasFocus, row, column, renderer);
    return renderer;
    //
  } // getTableCellRendererComponent


  /* *** *** */


  /**
   * Process command strings.
   * <p><pre>
   * process a command sent by e.g. the GUI menu system:
   *
   * showpart 12-34 (show that glyph range, using PART bitmap).
   * showall 12-34 (show that glyph range, using ALL bitmap).
   * jumpto 1234 (scrolls to that glyph).
   * jumpfield CHAR (scrolls to that glyph - char may be escaped).
   * hideunmapped glyphs (hide all unmapped glyphs from the table).
   * (hideunmapped, showall and showpart also use the checked argument)
   * </pre></p>
   * @param modifiers Modifiers use the normal ActionEvent bit masks.
   * @param checked Some commands use this as additional boolean input.
   */
  public void processCommand(String command, int modifiers, boolean checked) {
    // modifiers are 0 and 16 for keyboard and Lmouse, but other bits
    // can be set for shift, ctrl, alt, Mmouse, Rmouse. Not yet used.
    String action;
    String pos1s = "0";
    // if MapTable itself already parses commands:
      // super.processCommand(command, modifiers, checked);
    renderman.processCommand(command, modifiers, checked);
      // renderman handles jumpto and jumpfield.
    try {
      java.util.StringTokenizer strTok =
        new java.util.StringTokenizer(command);
      // default tokenizer, using " \r\n\t" as delimiters
      if (!strTok.hasMoreTokens()) return; // need command
      action = strTok.nextToken();
      if (!strTok.hasMoreTokens()) return; // need at least one argument
      pos1s = strTok.nextToken(" \t\n\r\f-"); // get first argument
        // second argument can be separated by "-" or space
        // pos2s = strTok.nextToken() - not needed HERE.
      if (action.equals("hideunmapped")) {
        //
        int minWidth = 50;
        int prefWidth = 75;
        int maxWidth = 100;
        if (myTable != null) {
          minWidth = myTable.getColumnModel().getColumn(0).getMinWidth();
          maxWidth = myTable.getColumnModel().getColumn(0).getMaxWidth();
          prefWidth = myTable.getColumnModel().getColumn(0).
            getPreferredWidth();
        }
        visor.processCommand(command, modifiers, checked);
          // causes recalculation list of visible rows
        this.fireTableStructureChanged(); // cause header repaint
        if (myTable != null) {
          myTable.getColumnModel().getColumn(0).setMinWidth(minWidth);
          myTable.getColumnModel().getColumn(0).setMaxWidth(maxWidth);
          myTable.getColumnModel().getColumn(0).
            setPreferredWidth(prefWidth);
          // restore width settings. sigh.
        }
        //
      } // hideunmapped
      else if (action.equals("showall") || action.equals("showpart")) {
        //
        visor.processCommand(command, modifiers, checked);
          // may cause the way that the table is displayed to
          // change (triggers updateVisibility() there), so:
        this.fireTableDataChanged(); // repaint everything...
        this.fireTableRowsUpdated(0, visor.getVisibleRowCount() - 1);
          // really repaint them!
        //
      } // showall or showpart
      else {
        //
        return; // ignore all commands that are not for us!
        //
      } // ignored commands
      //
      DebugEditIM.println(1, command + ", " + (checked ? "" : "un")
        + "checked, " + "modifiers=" + modifiers);
    } catch (Exception exc) {
      DebugEditIM.println(0, "error in command: " + command
        + " [modifiers=" + modifiers + ", checked=" + checked + "]");
      if (!command.startsWith("jumpfield")) // if not user fault...
        exc.printStackTrace();
      // no fatal error, just ignore the malformed command and continue
    } // catch
  } // processCommand


} // public class GlyphTable extends AbstractTableModel

