/**
 * Helper for the EditIM Input Method Editor:
 * Help and about window functions.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import guk.editIM.MenuHelpers;
import java.awt.*;
import java.awt.event.KeyEvent;
import javax.swing.*;
import javax.swing.event.*;



/**
 * Help and about window functions for the Input Method Editor.
 */
public class HelpEditIM {


 /**
  * The help window (accessed both for opening and closing it)
  * ... and the about window (same idea).
  */
 JFrame helpDialog = new JFrame("Help Window");
 JFrame aboutDialog = new JFrame("About Window");


 /**
  * A MenuHelpers object, containing listener and centering settings.
  */
 MenuHelpers menu = new MenuHelpers(null);


 /**
  * You have to hand over a MenuHelpers instance to the constructor.
  * @param menuHelper A MenuHelpers instance, used to create buttons
  * and center windows. Should have the listener set up correctly.
  */
 public HelpEditIM(MenuHelpers menuHelper) {
   if (menuHelper != null) menu = menuHelper;
   // if no menuHelper given, use a default one:
   // centering will not happen and the close
   // buttons will not work.
 }


 /**
  * Just shows the about window and some generic system properties.
  * Shows a typical about window and some system properties like
  * Java version of JRE and compiler, operating system and working
  * directory. The user can leave the window open while continuing
  * to work with other parts of the Input Method Editor. The window
  * is an HTML windows and following hyperlinks is possible.
  * @param openAbout Opens the window when true, otherwise closes it.
  */
 public void aboutWindow(boolean openAbout) {
   if (!openAbout) { // close help?
     aboutDialog.dispose();
     return;
   }
   // else open new help dialog:
   if (aboutDialog != null) aboutDialog.dispose();
   aboutDialog = new JFrame("About Input Method Editor"); // re-init
   //
   // could use a simple HTML JPanel, but a JEditorPane can be surfable
   // Java implements most of HTML 3.2 but does not like uppercase tags.
   String NL = "<br>";
     // JLabels do not understand br properly, only p. JEditorPane is okay.
   String aboutMessage1 = "<html><body bgcolor=\"#EEEEFF\" text=black>" +
      "<center>" + "<p><big><b>Input Method Editor</b></big></p>" +
      "<p>for editing GATE GUK, MPI EUDICO and" + NL + "<a href=" +
      "\"http://www.yudit.org/\">Yudit</a> Input Method keymaps." + NL +
      "Made at <a href=\"http://www.mpi.nl/\">MPI Nijmegen, NL</a>" +
      " in 2003 by Eric Auer" + NL +
      "Also <b>uses</b> <a href=\"http://gate.ac.uk/\">" +
      "GATE Unicode Kit</a> Input Methods" + NL + "</p></center>" +
      "<p><center><b>Runtime status:</b></center></p>";
   String aboutMessage2 = "";
   try {
     aboutMessage2 =
        "<p><b>Java VM " +
        System.getProperty("java.vm.specification.version") + "</b> [" +
        System.getProperty("java.vm.version") + "]" + NL + "<b>JRE " +
        System.getProperty("java.specification.version") + "</b> [" +
        System.getProperty("java.version") + "]" + NL + "System type: <b>" +
        System.getProperty("os.name") + " [" +
        System.getProperty("os.arch") + "] " +
        System.getProperty("os.version") + "</b>" + NL +
        "<b>extdirs:</b> " + System.getProperty("java.ext.dirs") + NL +
        "<b>classpath:</b> " + System.getProperty("java.class.path") + NL +
        "<b>User: " + System.getProperty("user.name") + "</b>" + NL +
        "<b>HOME:</b> " + System.getProperty("user.home") + NL +
        "<b>CWD: </b> " + System.getProperty("user.dir") + "</p>";
   } catch (Exception ex) {
     aboutMessage2 = "<p>(permission?) error reading status</p>";
   }
   // Object aboutMessage =
   //   aboutMessage1 + aboutMessage2 +  "</body></html>";
   JEditorPane aboutMessage = new JEditorPane("text/html",
     aboutMessage1 + aboutMessage2 + "</body></html>");
   aboutMessage.setCaretPosition(0); // cursor to the top!
   aboutMessage.setEditable(false);
   aboutMessage.addHyperlinkListener(new Hyperactive(aboutMessage));
   JScrollPane jSAM = new JScrollPane(aboutMessage);
   jSAM.setSize(new Dimension(400,350));
   jSAM.setMinimumSize(new Dimension(400,350));
   jSAM.setPreferredSize(new Dimension(400,350));
   aboutDialog.getContentPane().setLayout(new BorderLayout());
   aboutDialog.getContentPane().add(jSAM, BorderLayout.CENTER);
   aboutDialog.getContentPane().add(menu.myJButton(
     "Close about window", KeyEvent.VK_C,
     "Click here to close this about window", "showwindow aboutdone"),
     BorderLayout.SOUTH);
   aboutDialog.pack();
   aboutDialog.setVisible(true); // show();
   menu.centerMe(aboutDialog);
 } // aboutWindow


 /**
  * Just shows the help window.
  * The help window contains an HTML file and allows to browse
  * it (for example follow hyperlinks). The file is the resource
  * <a href="data/InputMethodEditorHelp.html">
  * data/InputMethodEditorHelp.html</a>. There is no BACK function
  * so be careful which hyperlinks you follow. You can always return
  * to the standard help file by closing and re-opening the window.
  * The user can leave the window open while continuing
  * to work with other parts of the Input Method Editor.
  * @param openHelp Opens the window when true, otherwise closes it.
  */
 public void helpWindow(boolean openHelp) {
   if (!openHelp) { // close help?
     helpDialog.dispose();
     return;
   }
   // else open new help dialog:
   if (helpDialog != null) helpDialog.dispose();
   helpDialog = new JFrame("Input Method Editor Help"); // re-init
   //
   JEditorPane helpPane = new JEditorPane(
     "text/html","<html><head><title>Error</title></head><body>"
     + "The Input Method Editor Help could not be loaded</body></html>");
   try {
     java.io.InputStream helpStream = this.getClass().
       getResourceAsStream("data/InputMethodEditorHelp.html");
     helpPane.read(helpStream,
       new javax.swing.text.html.HTMLDocument());
   } catch (Throwable th) {
     DebugEditIM.println(0,"Error reading HTML help resource.");
     // th.printStackTrace();
   }
   helpPane.setCaretPosition(0); // cursor to the top!
   helpPane.setEditable(false);
   helpPane.addHyperlinkListener(new Hyperactive(helpPane));
   helpDialog.getContentPane().setLayout(new BorderLayout());
   JScrollPane helpScrollPane = new JScrollPane(helpPane);
   helpDialog.getContentPane().add(helpScrollPane,
     BorderLayout.CENTER);
   helpDialog.getContentPane().add(menu.myJButton(
     "Close help window", KeyEvent.VK_C,
     "Click here to close this help window", "showwindow helpdone"),
     BorderLayout.SOUTH);
   helpDialog.pack();
   helpDialog.setSize(new Dimension(450, 600));
   helpDialog.setVisible(true); // show();
   menu.centerMe(helpDialog);
   helpScrollPane.getViewport().setViewPosition(new Point(1, 1));
     // does not help, as the HTML reading continues and the cursor
     // seems to be at the end... so the help scrolls down on open!
     // The setCaretPosition(0) above, however, does help.
  } // helpWindow


} // class HelpEditIM


///////////////////////////////////////////////////////////////////////////


 /**
  * taken from an example in the Java Swing documentation
  * (otherwise, I would not have been handling frames here...).
  * Added some ToolTips with current / touched URL myself.
  */
 class Hyperactive implements HyperlinkListener {


   /**
    * The component which bears our ToolTip.
    */
   JComponent compo = null;
   /**
    * The currently displaying URL.
    */
   String lastURL = "(local document)";


   /**
    * The constructor needs to know a target, so that the
    * object can update the target ToolTip when the user
    * clicks on or points at an hyperlink.
    * @param target Something that can have a ToolTip.
    */
   public Hyperactive(JComponent target) {
     compo = target;
     compo.setToolTipText("this tool tip shows the hyperlink status");
   } // constructor


   /**
    * Whenever the user clicks on, points at, or stops to point at
    * an hyperlink, this is triggered.
    * @param e The hyperlink event which tells us which URL the
    * user touched or triggered and what he did with that hyperlink.
    */
   public void hyperlinkUpdate(HyperlinkEvent e) {
     if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
       JEditorPane pane = (JEditorPane) e.getSource();
       if (e instanceof javax.swing.text.html.HTMLFrameHyperlinkEvent) {
         javax.swing.text.html.HTMLFrameHyperlinkEvent evt =
           (javax.swing.text.html.HTMLFrameHyperlinkEvent) e;
         javax.swing.text.html.HTMLDocument doc =
           (javax.swing.text.html.HTMLDocument)pane.getDocument();
         compo.setToolTipText("Last frame hyperlink: " + evt.getURL());
         lastURL = compo.getToolTipText();
         doc.processHTMLFrameHyperlinkEvent(evt);
       } // frame
       else {
         try {
           pane.setPage(e.getURL());
           compo.setToolTipText("Current URL: " + e.getURL());
           lastURL = compo.getToolTipText();
         } catch (Throwable t) {
           DebugEditIM.println(0,"Error following HTML hyperlink");
           // t.printStackTrace();
         }
       } // nonframe
     } // activated
     else if (e.getEventType() == HyperlinkEvent.EventType.ENTERED) {
       compo.setToolTipText("Pointer above link to: " + e.getURL());
     } // entered
     else if (e.getEventType() == HyperlinkEvent.EventType.EXITED) {
       compo.setToolTipText(lastURL);
     } // exited
   } // hyperlinkUpdate


 } // Hyperactive

