/**
 * Makes use of the Swing (Jstuff :-)) HTML highlighting for
 * JLabel, JButton, etc., by translating a key sequence string
 * to something with HTML markup inside.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */

/**
 * <p><pre>
 * KeyStroke creates keystrokes from strings like this. Notice that you
 * can have zero modifiers and none of pressed or released, leading to
 * possible strings like INSERT (and nothing else), too...
 *
 *     &lt;modifiers&gt;* (&lt;typedID&gt; | &lt;pressedReleasedID&gt;)
 *     modifiers := shift | control | meta | alt | button1 | button2 | button3
 *     typedID := typed &lt;typedKey&gt;
 *     typedKey := string of length 1 giving unicode character.
 *     pressedReleasedID := (pressed | released)? key
 *  key := KeyEvent keycode name, i.e. the name following &quot;VK_&quot;.
 *
 * Stupid enough, KeyEvent cannot GENERATE strings that match the ones
 * needed by KeyStroke, so we should write a nice KeyEvent to string
 * implementation that DOES handle that issue as wanted.
 * </pre></p>
 */

package guk.editIM;

import guk.editIM.*;

/**
 * Used to add HTML highlighting to strings that describe
 * keystrokes. Most really interesting keystroke stuff is
 * in AssignObject.java, not in this class!
 */
public class KeyStrokeParser {

  /**
   * Just a string buffer where the HTML is collected.
   */
  StringBuffer buf = new StringBuffer();
  /**
   * The font that will be mentioned in the HTML code.
   * Default value is Arial Unicode MS.
   */
  String fontName = ""; // can be overridden

  /**
   * The constructor does nothing, not needed to do anything.
   */
  public KeyStrokeParser() {
    // do init, if needed
  } // KeyStrokeParser

  /**
   * This constructor actually DOES something: It sets the font.
   * @param font The font that will be mentioned in the HTML code.
   */
  public KeyStrokeParser(java.awt.Font font) {
    if (font != null) {
      fontName = font.getName();
    } else {
      fontName = ""; // if null font for constructor
    }
  } // KeyStrokeParser

  /**
   * highlightComplete uses HTML markup and adds HTML header / footer.
   * @param what A string that describes key strokes.
   * @return A string with HTML that contains the input string and
   * additional highlighting tags, starting with &lt;html&gt; and
   * containing all headers and footers and an optional font selection.
   * <b>Note that JLabels only understand HTML 3.2 with font color
   * and size, but not font name. They cannot do Unicode in HTML mode
   * either, as of Java 1.3.1 ...</b>
   */
  public String highlightComplete(String what) {
    return "<html>" + ((fontName.length() > 0) ?
        "<font name=\"" + fontName + "\">" : "")
      + "<pre>" + highlight(what) + "</pre>"
      + ((fontName.length() > 0) ? "</font>" : "") + "</html>";
    // MUST use Font name , or default font will be used!
    // pre prevents wrapping, but causes ugly "typewriter" spacing :-(
  } // highlightComplete

  /**
   * highlight uses HTML markup to emphasize some special
   * escapes in the input string. Also escapes HTML reserved
   * chars and Unicode <b>(Unicode needs HTML 4.0 to render,
   * JLabels only render ISO-8859-1 HTML 3.2)</b>
   * Recognized escapes:
   * <ul>
   * <li>&amp;, &lt;, &gt; and space are translated to HTML
   *   (space into <u>&nbsp;</u> to prevent wrapping)</li>
   * <li><i>C-</i>, <i>A</i>-, ... prefixes:
   *   ctrl, alt, ..., \- means - (no prefix)</li>
   * <li><b><tt>\\uxxxx</tt></b> Unicode char number xxxx
   *   (4 hex digits)</li>
   * <li><tt><u>\n</u></tt>, <tt><u>\r</u></tt>, <tt><u>\\</u></tt>
   *   as known from C and Java</li>
   * <li><b>\\k+xxxx+</b>: a string for
   *   getKeyStroke(&quot;xxxx&quot;)</li>
   * </ul>
   * see Classes KeyStroke and KeyEvent, too. + can be any char.
   * notice that the contents of \., \k+...+, \\u.... and .- are
   * not parsed here: we only use some knowledge about their length.
   * @param what A string that should be highlighted.
   * @return The string with added HTML tags for highlighting, but
   * without headers or footers like &lt;html&gt;...
   */
  public String highlight(String what) {
    buf = new StringBuffer(what.length() + 13);
    // could simply return what itself if we detect
    // Java below version 1.3 - not checked, as the
    // user may have other HTMLization reasons.
    char [] str = what.toCharArray(); // array of chars in the string
    char delim = ' '; // delimiter for \k escapes
    boolean kactive = false; // is a \k escape active?
    int i = 0;
    while (i < str.length) {
      char here = str[i];
      i++;
      if (here > (char)125) {
        buf.append("&#" + (int)here + ";");
      }
      else if (here == '<') { buf.append("&lt;"); }
      else if (here == '>') { buf.append("&gt;"); }
      else if (here == '&') { buf.append("&amp;"); }
      else if (here == ' ') { buf.append("<u>&nbsp;</u>"); }
      // ... else handled at various places BELOW
        // we did not "continue;" above as the "delim" could be
        // the same char as "here"!
      if (kactive) { // do not escape other stuff inside \k escapes
        if ((here <= (char)125) && (here != '<') && (here != '>') &&
            (here != '&') && (here != ' ')) // if not handled ABOVE
          buf.append(here); // belongs to \k escape
        if (here == delim) { // end of \k escape?
          kactive = false;
          buf.append("</b></u>"); // close \k highlighting
        } // kactive end
        continue;
      } // kactive mode
      if ((here == '<') || (here == '>') || (here > (char)125) ||
          (here == '&') || (here == ' ')) // if handled ABOVE
        continue; // finally do the continue which we were not
          // able to do before making sure to handle kactive
      if (i == str.length) { // last char in string
        buf.append(here); // last char
        continue; // skip to next iteration
      }
      // only 2 char stuff from here on
      char next = str[i];
      i++;
      if ((next == '-') && (here != '\\')) { // shifty prefix
        buf.append("<i>");
        buf.append(here); // not translated, accepts all
        buf.append("-</i>"); // not highlighting the thing itself,
          // only the prefixes (easier)
      } // C- A- and similer
      else if (here == '\\') { // all \... escapes
        if (next == 'u') { // unicode escape
          if ((i+4) < str.length) { // VALID unicode escape
            buf.append("<b><tt>\\");
            buf.append(next);
            buf.append(str[i]); i++;
            buf.append(str[i]); i++;
            buf.append(str[i]); i++;
            buf.append(str[i]); i++;
            buf.append("</tt></b>");
          } else { // invalid unicode escape
            buf.append("<u>\\u</u>");
          }
        } else if (next == 'k') { // keystroke escapes
          if (i != str.length) {
            delim = str[i];
            i++;
            buf.append("<b>\\k");
            buf.append(delim); // remember delimiter
            kactive = true; // activate kactive mode
          } // (... else invalid keystroke escape)
        } else { // single escaped char
          buf.append("<tt><u>\\");
          buf.append(next);
          buf.append("</u></tt>");
        }
      } // all \... escapes
      else { // normal text
        buf.append(here);
        i--; // rewind, as "next" is not used.
      } // normal text
    } // while
    if (kactive) buf.append("</b></u>"); // close \k highlighting
    kactive = false;
    return buf.toString();
  } // highlight

} // class KeyStrokeParser

