/**
 * An abstract implementation of the data model for the unicode
 * keymap editor tables in terms of an AbstractTableModel.
 * Also contains renderer and editor code, note the unusual
 * place! Allowing them to access internal data that way.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;


import guk.editIM.DebugEditIM;
import javax.swing.table.*; // table models and related
import javax.swing.*; // JTable and similar
import java.awt.*; // color stuff, renderer stuff
import java.util.Vector;
import java.util.List;
import java.util.Collection;



/**
 * An abstract implementation of the data model for the unicode
 * keymap editor tables in terms of an AbstractTableModel.
 * Contains additional methods for controlling the data and
 * importing and exporting it in form of AssignObject objects.
 * Also contains renderer and editor code, note the unusual
 * place! Allowing them to access internal data that way.
 */
public abstract class MapTable extends AbstractTableModel
  implements TableCellRenderer {


  /**
   * A standard renderer which we tune below.
   */
  protected DefaultTableCellRenderer tcRenderer = new DefaultTableCellRenderer();


  /**
   * Format to be used by getValueAt and setValueAt
   * (and therefore by Cell Editors, too!).
   */
  protected final int transferFormat = AssignObject.UNICODE_HUMAN;



  /**
   *  The table data, can grow on demand using our addRow() method.
   *  A flag tells when setValueAt modified the data (other kinds
   *  of modfication do not change the flag).
   */
  protected Vector assignments = new Vector(1,16);
  protected boolean changedFlag;


  /**
   * The default constructor does not do much.
   */
  public MapTable() {
    // start with empty table
  } // constructor


  /**
   * Removes all entries, returns false on failure
   * @return The standard implementation always succeeds,
   * but implementing classes can obviously override that behaviour.
   */
  public boolean flushTable() {
    assignments = new Vector(1, 16);
    changedFlag = false;
    return true;
  } // flushTable


  /**
   * Returns the Collection with all entries (for saving them).
   * @return A Collection of AssignObject objects. It is not
   * recommended to modify the returned objects or remove some
   * of them, although some implementations may be able to
   * handle that.
   */
  public Collection readTable() {
    return assignments;
  } // readTable


  /**
   * Merges an entry into the table, returns false on failure
   * Normally, merge succeeds, but in the GlyphTable implementation,
   * further constraints have to be followed.
   * @returns True unless something went wrong and the data could
   * not be merged into the table.
   */
  public boolean addEntry(AssignObject ao) {
    if (ao == null) return false;
    /**
     * @todo: could update some features-in-use flags here,
     * in order to allow more specialized warnings before
     * writing to a file.
     */
    assignments.add(ao);
    return true;
  } // addEntry


  /**
   * Finds and removes an entry from the table, returns false on failure
   * @return True if removal succeeded. Failure is usually because the
   * object that you tried to remove was not found in the table.
   */
  public boolean removeEntry(AssignObject ao) {
    if (ao == null) return false;
    return assignments.remove(ao);
  } // removeEntry


  /**
   * Checks if the table contains unsaved changes.
   * @return True if the table has been modified by setValueAt
   */
  public boolean isChanged() {
    return changedFlag;
  } // isChanged


  /**
   * Clear change flag. Tells the table that there are no
   * unsaved changes (anymore).
   */
  public void clearChanged() {
    changedFlag = false;
  } // clearChanged


  /* *** *** */


  /**
   * Checks how many rows the table contains.
   * This must be efficient as Java calls it often.
   * @return The number of rows in the table.
   */
  public int getRowCount() {
    return assignments.size(); // must be efficient!
  }


  /**
   * Checks how many columns the table contains.
   * Always 3: Glyphs, keys and comments.
   * @return The value 3.
   */
  public int getColumnCount() {
    return 3; // a glyph string column
      // and one for keymap trigger strings
      // and one for comments
  }


  /**
   * Fetch the contents of a certain table cell.
   * Contents will be converted into a strings for easier use.
   * The transferFormat object variable determines the format
   * of the string (usually human readable Unicode).
   * @param row The table row, selecting an AssignObject.
   * @param col The column, selecting the part of the AssignObject.
   * @return The object found at the given place in the table,
   * which will be something that can be displayed.
   */
  public Object getValueAt(int row, int col) {
    if ( (row < 0) || (row >= getRowCount()) ) {
      DebugEditIM.println(0, "getValueAt(" + row + ", " + col + ") ??");
      // return null;
      return new String("hello there?");
    }
    if (col == 0) return
      ((AssignObject)assignments.elementAt(row)).
      exportGlyphs(transferFormat);
    if (col == 1) return
      ((AssignObject)assignments.elementAt(row)).
      exportKeys(transferFormat);
    if (col == 2) return
      ((AssignObject)assignments.elementAt(row)).
      getComments();
    return null;
  }


  /* *** *** */


  /**
   * Fetches the column names for the headers and for
   * finding columns.
   * @param col The column for which you want to know the name.
   * @return &quot;Glyph Strings&quot;,
   * &quot;Key Sequences&quot; or &quot;Comments&quot; for
   * columns 0, 1 and 2 respectively.
   */
  public String getColumnName(int col) {
    if (col == 0) return "Glyph Strings";
    if (col == 1) return "Key Sequences";
    if (col == 2) return "Comments";
    return "???";
  } // getColumnName


  // findColumn is not overridden, use default which compares headers.


  /**
   * Fetches the column data types, which Java uses to select
   * a renderer component for the column.
   * @param col The column for which you want to know the type.
   * @return String (all columns return String type data when
   * you call getValueAt).
   */
  public Class getColumnClass(int col) {
    if (col == 0) return new String("hello").getClass();
    if (col == 1) return new String("hello").getClass();
    if (col == 2) return new String("hello").getClass();
    return new Object().getClass();
  }


  /**
   * Tells whether a certain cell can be edited.
   * @param row The affected row (not likely to matter)
   * @param col The affected column: implementing classes
   * may return false for columns where they do not want Java
   * to allow the user to activate the editor component.
   * @returns true (implementing classes can override this).
   */
  public boolean isCellEditable(int row, int col) {
    // ignores row, all rows are treated the same here
    if (col == 0) return true; // glyph sequences are also editable!
    if (col == 1) return true; // assigned sequences are editable
    if (col == 2) return true; // comments are editable
    return false;
  }


  /**
   * Update a cell value. Sets the changedFlag.
   * This is where all editing results come in. The user can edit
   * cells, and when he commits the results, they end up here.
   * Notice that this cannot fail, apart from ignoring senseless
   * updates. The user will notice the non-change and try the edit
   * again, but the erroneous value is lost.
   * @param val Usually a string with the new value for that cell.
   * @param row The row where the value is to be stored.
   * @param col The column where the value is to be stored.
   */
  public void setValueAt(Object val, int row, int col) {
    /** *** would update running GUK IM FSM here *** */
    if (!(val instanceof String)) return; // ignore other attempts!
    if (col == 0) { // glyphs column
      String old = ((AssignObject)assignments.elementAt(row)).
        exportGlyphs(transferFormat);
      ((AssignObject)assignments.elementAt(row)).
        importGlyphs((String)val, transferFormat);
      changedFlag = true; // could check for ACTUAL change here
    } // glyphs column
    if (col == 1) { // key strings column
      String old = ((AssignObject)assignments.elementAt(row)).
        exportKeys(transferFormat);
      ((AssignObject)assignments.elementAt(row)).
        importKeys((String)val, transferFormat);
      changedFlag = true; // could check for ACTUAL change here
    } // key strings column
    if (col == 2) { // comments column
      String old = ((AssignObject)assignments.elementAt(row)).
        getComments();
      ((AssignObject)assignments.elementAt(row)).
        setComments((String)val);
      changedFlag = true; // could check for ACTUAL change here
    } // key strings column
    /**
     * would update running GUK IM FSM here (replace result on output,
     * which is possibly just one of several choice list items)
     */
    this.fireTableCellUpdated(row, col); // repaint etc.
  }


  // addTableModelListener and
  // removeTableModelListener and
  // getTableModelListeners and
  // getListeners not overridden, using default



  // fireTableDataChanged (all data changed) and
  // fireTableStructureChanged (column name/type/number changed) and
  // fireTableRowsInserted(first, last) (...) and
  // fireTableRowsUpdated(first, last) (...) and
  // fireTableRowsDeleted(first, last) (...) and
  // fireTableCellsUpdated(row, column) and
  // fireTableChanged(TableModelEvent e) (user-defined / ...)
  //   not overridden, but USED by this class. setValueAt
  //   does not need to fire TableCellsUpdated, by the way.



  /* *** *** */


  /**
   * Add a row to the table.
   * This special MapTable method adds a row to the current end
   * of the table. The row will contain some empty AssignObject.
   */
  public void addRow() { // allow the world out there to request expansion
    // strings.add("");
    // glyphs.add("");
    assignments.add(new AssignObject());
    this.fireTableRowsInserted(assignments.size()-1, assignments.size()-1);
  } // addRow


  /* *** *** */


  /**
   * Returns a component that renders the given cell. Note that
   * table models usually do NOT care for the related looks, but
   * MapTable objects provide this method, which you can hand over
   * to the table that renders the MapTable.
   * <p>
   * This table model provides its own renderer, which sets the
   * color according to which unicode range holds the affected row
   * </p><p><i>
   * WARNING: TableModels normally do not implement TableCellRenderer
   * and this exception (activated via table.setDefaultRenderer(...))
   * means that we have to be CAREFUL with coordinates: JTable and
   * renderer stuff uses VIEW coordinates, but the Model uses MODEL
   * coordinates. The user may have swapped (by dragging) the columns!
   * </i></p>
   * @see javax.swing.table.TableCellRenderer
   */
  public Component getTableCellRendererComponent(
    JTable table, Object value, boolean isSelected,
    boolean hasFocus, int row, int column) {
    int i;
    // table can be null, row can be -1 for the header
    // this is meant to configure the renderer to draw the
    // next cell, as described in the parameters.
    Component aComponent = tcRenderer.getTableCellRendererComponent(
      table, value, isSelected, hasFocus, row, column);
    if (!(aComponent instanceof JLabel)) {
      DebugEditIM.println(2,
        "cannot beautify table cell: render Component is no JLabel");
      return aComponent;
    }
    JLabel theLabel = (JLabel)aComponent;
    theLabel.setHorizontalTextPosition(JLabel.LEFT);
    theLabel.setHorizontalAlignment(JLabel.LEFT);
    //
    int modelColumn = table.convertColumnIndexToModel( column );
    //
    if (modelColumn == 0) {
      theLabel.setToolTipText("Enter Unicode text (output)");
    } else if (modelColumn == 1) {
      theLabel.setToolTipText("Enter text and/or keystrokes (input)");
    } else if (modelColumn == 2) {
      theLabel.setToolTipText("Enter (free-form) comments");
    }
    if ( (!(value instanceof String)) || (row >= assignments.size()) )
      return theLabel; // we do not know enough to HTMLize
      // (probably, a subclass uses us but does not use assignments)
    //
    AssignObject ao = (AssignObject)(assignments.elementAt(row));
    if (modelColumn == 1) {
      if ( !ao.getKeys().isEmpty() ) {
        String htmlValue = ao.exportKeys(AssignObject.UNICODE_HTML);
          // from Java 1.3 on, JLabels and other JStuff accepts HTML!
        if (htmlValue.indexOf("&#") > 0) {
          // this is a bit oversimplified, but better more Unicode
          // than more highlighting. We would check for non ISO-8859-1
          // rather than for ANY &amp;#1234; to be more correct.
          DebugEditIM.println(4, "not HTMLizing: Unicode in use!");
          // leaving value as is.
        } else {
          value = htmlValue;
          DebugEditIM.println(5, "HTMLized: " + value);
        }
      } else {
        value = ""; // save time: no HTML needed to say nothing.
      }
    } // Keys that we know about
    else if (modelColumn == 0) {
      value = ao.getGlyphs();
        // value hopefully already had that value anyway, but we want
        // to make sure that row has the same meaning in both columns.
        // (we do not use HTML in Glyphs, not useful)
    } // Glyphs that we know about
    else if (modelColumn == 2) {
      value = ao.getComments();
        // value hopefully already had that value anyway, but we want
        // to make sure that row has the same meaning in both columns.
    } // Comments that we know about
    //
    theLabel.setText((String)value);
    return theLabel;
    //
  } // getTableCellRendererComponent


  /* *** *** */


  /**
   * The generic MapTable does not process any commands yet:
   * Implementing classes are likely to override this method,
   * so that you can send them commands.
   * @param command A textual representation of something that
   * should happen with the table data.
   * @param modifiers ActionEvent modifiers which may affect
   * the command.
   * @param checked A boolean value that may affect the command.
   */
  public void processCommand(String command, int modifiers, boolean checked) {
    // from the GUI menu: addchar row glyph (append glyph to glyphs[row])
    // modifiers are 0 and 16 for keyboard and Lmouse, but other bits
    // can be set for shift, ctrl, alt, Mmouse, Rmouse. Not yet used.
    String action;
    char glyph;
    int row = 0;
    try {
      java.util.StringTokenizer strTok =
        new java.util.StringTokenizer(command);
      // default tokenizer, using " \r\n\t" as delimiters
      if (!strTok.hasMoreTokens()) return; // need command
      action = strTok.nextToken();
      // would do something with the command here
      DebugEditIM.println(3, "no parser installed, ignored command: "
        + command);
    } catch (Exception exc) {
      DebugEditIM.println(0, "error in command: " + command
        + " [modifiers=" + modifiers + ", checked=" + checked + "]");
      // no fatal error, just ignore the malformed command and continue
    }
  } // processCommand


} // public class MapTable extends AbstractTableModel


