/*
 * MenuHelpers.java in guk.editIM: Simplify creation of buttons,
 * menu items, checkboxes... Other menu related tasks.
 * By Eric Auer 2003.
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */


package guk.editIM;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;


/**
 * MenuHelpers.java in guk.editIM: Simplify creation of buttons,
 * menu items, checkboxes... Other menu related tasks.
 */
public class MenuHelpers {


  /**
   * Going through the event queue is broken.
   * After fixing it, enable it here.
   */
  final boolean REALLYDISPATCH = false;


  /**
   * The action listener that will receive all events from menu
   * elements generated by this class.
   */
  ActionListener ACTIONLISTENER = null;


  /**
   * The font which is used by the setSyncFont and syncFont methods.
   */
  Font theFont = new Font(null /* default */, 0 /* plain */, 10);


  /**
   * The component relative to which centerMe will center things.
   */
  Component baseWin = null;


  /**
   * Tell the constructor what the action listener should be.
   * @param listener The action listener that will receive all
   * events from menu elements generated by this class.
   */
  public MenuHelpers(ActionListener listener) {
    ACTIONLISTENER = listener;
  } // constructor


  /**
   * Use a new action listener in the future.
   * @param listener The action listener that will receive all
   * events from menu elements generated here in the future.
   */
  public void setListener(ActionListener listener) {
    ACTIONLISTENER = listener;
  } // setListener


  /**
   * Retrieve the current action listener value.
   * @return The action listener that receives all
   * events from menu elements generated by this class.
   */
  public ActionListener getListener() {
    return ACTIONLISTENER;
  } // getListener


  /**
   * Dispatches either by dispatchEvent or by calling
   * actionPerformed directly if dispatchEvent is not
   * available. The former is better as it does not
   * mess with the queueing and timeline.
   * @param target The target, a Component or
   * at least an ActionListener.
   * @param event An ActionEvent (not AWTevent)
   * or command String to be sent.
   */
  void posting(Object target, Object event) {
    ActionEvent what = null;
    if (event instanceof ActionEvent) {
      what = (ActionEvent)event;
    } else if (event instanceof String) {
      what = new ActionEvent(this,
        ActionEvent.ACTION_PERFORMED, (String)event);
    } else {
      DebugEditIM.println(0, "You cannot send that: " + event);
    }
    DebugEditIM.println(4, target.getClass().getName()
      + " <-- " + what.getActionCommand());
    if (REALLYDISPATCH && (target instanceof Component)) {
      ((Component)target).dispatchEvent(what);
    } else {
      DebugEditIM.println(4, "bypassing queue!");
      try {
        ((ActionListener)target).actionPerformed(what);
      } catch (ClassCastException cce) {
        DebugEditIM.println(0, "Could not send event:\n"
          + event + "\nto target:\n" + target + "\n");
      } // try / catch
    } // target is no Component
  } // posting


  /**
   * Dispatch event to our listener, giving it
   * a new ActionEvent (with id ACTION_PERFORMED)
   * with the specified message.
   * @param The string that will be used to generate
   * the sent action event.
   */
  public void tellListener(String what) {
    posting(ACTIONLISTENER, what);
  } // tellListener


  /**
   * Dispatch event to ANOTHER listener, giving it
   * a new ActionEvent (with id ACTION_PERFORMED)
   * with the specified message.
   * @param whom The listener that should be notified.
   * @param what The string that will be used to generate
   * the sent action event.
   */
  public void tellListener(ActionListener whom, String what) {
    posting(whom, what);
  } // tellListener


  /**
   * Dispatch event to ANOTHER listener, using a given
   * ActionEvent.
   * @param whom The listener that should be notified.
   * @param what The ActionEvent to be sent.
   */
  public void tellListener(ActionListener whom, ActionEvent what) {
    posting(whom, what);
  } // tellListener


  /**
   * Enable the centering functionality.
   * @param compo The component relative to which centerMe
   * will center things
   */
  public void setBaseWin(Component compo) {
    baseWin = compo;
  } // setBaseWin


  /**
   * Ask for the current centering base component.
   * @return compo The component relative to which centerMe
   * will center things
   */
  public Component getBaseWin() {
    return baseWin;
  } // getBaseWin


  /**
   * Set up the font parameter for font syncing.
   * @param font An arbitrary font.
   */
  public void setSyncFont(Font font) {
    theFont = font;
  } // setSyncFont


  /**
   * Sync the font used by the given component: Set it
   * to the font selected by setSyncFont.
   * @param compo A component that you want to use the
   * font selected by setSyncFont.
   */
  public void syncFont(Component compo) {
    compo.setFont(theFont);
  } // syncFont


 /**
  * Generic creation of box, item or menu type menu items.
  * Helper to create a JMenuItem or JCheckBoxMenuItem or JMenu,
  * with all settings in one go. Values for what: box, item, or menu.
  * The action listener is always ACTIONLISTENER.
  * Initially, input methods are disabled for the component.
  * @param what can be "box", "item" or "menu" to select what
  * type of item (JCheckBoxMenuItem, JMenuItem, JMenu) we need.
  * @param label The text for the item. If null, a default is used.
  * @param mnemonic The key code that should be used as mnemonic.
  * Use -1 to indicate no mnemonic.
  * @param tooltip The tool tip text, if any (null otherwise).
  * @param command The action command that should be used. For
  * a menu, null should be used here.
  */
 public JMenuItem myJMenuItem(String what, String label,
   int mnemonic, String tooltip, String command) {
   JMenuItem item = null;
   if (label == null) label = "<item>";
   if (what.equals("item")) { item = new JMenuItem(label);
   } else if (what.equals("box")) { item = new JCheckBoxMenuItem(label);
   } else { item = new JMenu(label);
   }
   if (mnemonic >= 0) item.setMnemonic(mnemonic);
   if (tooltip != null) item.setToolTipText(tooltip);
   if (command != null) item.setActionCommand(command);
   item.enableInputMethods(false);
   item.addActionListener(ACTIONLISTENER);
   return item;
 } // myJMenuItem


 /**
  * Helper to create a mini button for special glyph functions
  * with all settings in one go.
  * The action listener is always ACTIONLISTENER.
  * Initially, input methods are disabled for the component.
  * @param label The text that the button should show
  * @param tooltip The tool tip for the button
  * @param command the action command for the button.
  */
 public JButton glyphXButton(String label, String tooltip,
   String command) {
   JButton xbutton = new JButton(label);
   xbutton.setToolTipText(tooltip);
   xbutton.setActionCommand(command);
   xbutton.addActionListener(ACTIONLISTENER);
   xbutton.setFont(new Font(null /* default */, 0 /* plain */, 10));
   xbutton.setMargin(new Insets(1,1,1,1)); // almost no border
   xbutton.enableInputMethods(false);
   return xbutton;
 } // glyphXButton


 /**
  * Helper to create a JButton with all settings in one go.
  * Input Methods will be initially disabled.
  * The action listener is always ACTIONLISTENER.
  * @param label The text for the item. If null, a default is used.
  * @param mnemonic The key code that should be used as mnemonic.
  * Use -1 to indicate no mnemonic.
  * @param tooltip The tool tip text, if any (null otherwise).
  * @param command The action command that should be used.
  */
 JButton myJButton(String label, int mnemonic, String tooltip,
   String command) {
   JButton button = new JButton(label);
   button.setMnemonic(mnemonic);
   button.setToolTipText(tooltip);
   button.setActionCommand(command);
   button.addActionListener(ACTIONLISTENER);
   button.setMaximumSize(new Dimension(200, Short.MAX_VALUE));
     // one size for all of our buttons
   button.enableInputMethods(false);
   return button;
 } // myJButton


 /**
  * Force minimum, maximum and preferred size of a
  * component to a certain value.
  * @param compo The component that will have the size
  * hints changed.
  * @param size The new value for the size hints.
  */
 static public void fixSize(JComponent compo, Dimension size) {
   compo.setPreferredSize(size);
   compo.setMinimumSize(size);
   compo.setMaximumSize(size);
 } // fixSize


 /**
  * Convert an int value into an hex String of
  * a length which is a multiple of 4.
  * @param value An integer value, preferrably 0..65535
  * @return A string representation of the argument, which
  * will be a multiple of 4 characters long and hexadecimal.
  */
 static public String toHex(int value) {
   String s = Integer.toHexString(value);
   if (s.length() == 0) return "0000";
   switch (s.length() & 3) {
     case 1: return "000" + s;
     case 2: return "00" + s;
     case 3: return "0"  + s;
     default: return s;
   } // switch
 } // toHex


 /**
  * Centers the given component relative to the baseWin component.
  * Use setBaseWin() to set the baseWin value.
  * @param compo A component that should be centered relative
  * to the component given to setBaseWin() (if none is known,
  * nothing will happen).
  */
  public void centerMe(Component  compo) {
    if (compo == null) return; // no centering possible
    if (baseWin == null) return; // no centering possible
    java.awt.Dimension dimBase = baseWin.getSize();
    java.awt.Dimension dimCompo = compo.getSize();
    double x = (dimBase.getWidth() - dimCompo.getWidth()) / 2;
    double y = (dimBase.getHeight() - dimCompo.getHeight()) / 2;
    // avoid getting too close to the upper left corner:
    if (x < 50) x = 50;
    if (y < 50) y = 50;
    compo.setLocation((int)x, (int)y);
    // setLocation is relative to parent coordinate system
    // (default: screen). We also have Point getLocation[OnScreen].
  } // centerMe


  /**
   * Helper method: modify hue in HSB color space by some
   * function of hash and by offset offs.
   * @param c input color, will be somewhat trimmed
   * @param hash arbitrary integer as seed for the hue modification
   * @param offs offset to the hue
   */
  static public Color hueByHash(Color c, int hash, float offs) {
    float [] hsb = new float [3]; // hue saturation brightness
      // hue is treated - floor(hue), so any value is okay.
    hsb = Color.RGBtoHSB(c.getRed(), c.getGreen(), c.getBlue(), hsb);
    hash ^= (hash >> 8) ^ (hash >> 16) ^ (hash >> 24);
    hsb[0] += offs;
    hsb[0] += (float)( (1.0/256) * (hash & 255) );
    hsb[1] = (float)Math.max(0.25, hsb[1]); // not too uncolored
    hsb[2] = (float)Math.max(0.25, hsb[2]); // not too dark
    return Color.getHSBColor(hsb[0], hsb[1], hsb[2]);
  } // hueByHash



} // class MenuHelpers

