/**
 * An implementation of the data model for the many to many unicode
 * keymap editor table in terms of an AbstractTableModel.
 * By Eric Auer 2003.
 *
 * This file is part of the Input Method Editor made at
 * http://www.mpi.nl/ and is free software, licensed under
 * the GNU General Public License (GPL) which can
 * be found at http://www.gnu.org/licenses/gpl.txt or in the
 * file EditIM-COPYING.txt included in this distribution.
 *
 * Note that some other EditIM files have LGPL license.
 * GPL means: You may copy, use and edit code (not license) at
 * your wish. Everything that contains GPLed code must be GPLed,
 * too. Sources must be available to all users of the binaries.
 * With GPL, you still have to provide access to THIS source
 * file, but the rest of your project can stay closed source.
 */

package guk.editIM;

import guk.editIM.*;
import javax.swing.table.*; // table models and related
import javax.swing.*; // JTable and similar
import java.awt.*; // color stuff, renderer stuff
import java.util.Vector;

/**
 * An implementation of the data model for the many to many unicode
 * keymap editor table in terms of an AbstractTableModel.
 * Most methods are explained in their MapTable versions already.
 * @see MapTable .
 */
public class StringTable extends MapTable
  /* which extends AbstractTable Model */
  implements TableCellRenderer {


  /**
   * The table initially contains a single demo object.
   */
  public StringTable() {
    AssignObject ao = new AssignObject();
    ao.setGlyphs("glyhps");
    ao.importKeys("C-kC-eC-yC-s", AssignObject.UNICODE_HUMAN);
    ao.setComments("example");
    assignments.add(ao);
  } // constructor


  /**
   * The there must be a row that the user can select, because
   * selecting the last row means adding a row to the StringTable
   * in the EditIM user interface.
   * @return The number of rows that exist in the table, never 0.
   */
  public int getRowCount() {
    int rows = super.getRowCount();
    if (rows != 0) return rows;
    this.addRow();
    return super.getRowCount();
  }


  /**
   * When setting the value leads to both key sequence and glyph
   * string becoming empty, the affected row will be removed.
   * @see MapTable#setValueAt(Object,int,int)
   * setValueAt(Object,int,int) in MapTable.
   */
  public void setValueAt(Object val, int row, int col) {
    super.setValueAt(val, row, col);
    if (  ((AssignObject)assignments.elementAt(row)).getKeys().isEmpty()
       && ((AssignObject)assignments.elementAt(row)).getGlyphs().equals("")
       ) { // if not mapped and empty, remove row.
       assignments.removeElementAt(row);
       this.fireTableRowsDeleted(row,row); // update LOOK
       DebugEditIM.println(1, "removed now empty row " + row);
    }
  } // setValueAt


  // the renderer is okay, we only change the editor


  /* *** *** */


  /**
   * This processes incoming commands.
   * addchar row glyph is the only command that StringTable accepts.
   * The addchar command will append a glyph to a row, without checking
   * if the glyph column in that row is currently being edited, so you
   * should do that before sending the command.
   * @see MapTable#processCommand(String,int,boolean)
   * processCommand(String,int,boolean) in MapTable.
   */
  public void processCommand(String command, int modifiers, boolean checked) {
    // from the GUI menu: addchar row glyph (append glyph to glyphs[row])
    // modifiers are 0 and 16 for keyboard and Lmouse, but other bits
    // can be set for shift, ctrl, alt, Mmouse, Rmouse. Not yet used.
    // if MapTable parses commands:
    // super.processCommand(command, modifiers, checked);
    String action;
    char glyph;
    int row = 0;
    try {
      java.util.StringTokenizer strTok =
        new java.util.StringTokenizer(command);
      // default tokenizer, using " \r\n\t" as delimiters
      if (!strTok.hasMoreTokens()) return; // need command
      action = strTok.nextToken();
      if (!action.equals("addchar")) return; // no command for us
      if (!strTok.hasMoreTokens()) return; // need two arguments
      row = Integer.parseInt(strTok.nextToken());
        // get first numeric argument
      if (!strTok.hasMoreTokens()) return; // need two arguments
      glyph = strTok.nextToken().charAt(0); // get second char argument
      DebugEditIM.println(1, action + ", " + row + ", " + glyph + ", "
        + (checked ? "" : "un") + "checked, modifiers=" + modifiers);
      //
      this.setValueAt( (String)(this.getValueAt(row, 0)) + glyph,
        row, 0 /* left column */);
        // FEEL update is done in setValueAt()
        // add glyph to the end of the string, other editing is done
        // simply using the default editor.
    } catch (Exception exc) {
      DebugEditIM.println(0, "error in command: " + command
        + " [modifiers=" + modifiers + ", checked=" + checked + "]");
      // no fatal error, just ignore the malformed command and continue
    }
  } // processCommand


} // public class StringTable extends MapTable

