/**
 * ShiftedKeys.java - 2003 by Eric Auer.
 * Provides a way to manage the looks of your keyboard with respect
 * to which key is the (un)shifted version of which other key.
 *
 * This file is part of GATE (see http://gate.ac.uk/), and is free
 * software, licenced under the GNU Library General Public License,
 * Version 2, June 1991.
 *
 * A copy of this licence is included in the distribution in the file
 * licence.html, and is also available at http://gate.ac.uk/gate/licence.html.
 */

package uk.ac.gate.guk.im;

import javax.swing.KeyStroke;
import java.awt.event.KeyEvent;
import java.lang.Character;

/**
 * Provides a way to manage the looks of your keyboard with respect
 * to which key is the (un)shifted version of which other key.
 * Only method is static xlate(char old, boolean shifted) which
 * translates a char into how the result of pressing the same key
 * with shift being in state shifted. Alternative method uses int old,
 * so that you can use VK_... keycodes, too.
 */
public class ShiftedKeys {

  /**
   * set KMAP to decide which keyboard type should be assumed
   * in the char version of xlate.
   */

  /**
   * Constant to select US keymap.
   */
  public static final int USMAP = -1;

  /**
   * Constant to select UK keymap. Mostly like US but has
   * one key more and uses the ', 2, 3 and upper left (accent)
   * keys differently. Pound and NOT sign are only on UK, not
   * US, and ` ~ @ # ' &quot; are on different positions.
   */
  public static final int UKMAP = 1;

  /**
   * Constant to select the DE (German) keymap. Quite a few
   * changes compared to UK: Shift numbers are used differently,
   * 3 Umlaut keys replace the [{ ;: '&quot; keys, shift ,. is
   * now ;:, &lt;&gt; have a new key, -_ key is moved, the key
   * next to Enter (UK style) is now #', ]} is now +*, and on the
   * old -_ =+ positions we have an &szlig;? and an accent key.
   * Several of the US
   * symbols therefore can only be reached with AltGr, which is
   * a new shift that replaces right Alt. Those are: @{[]}\|~.
   * Improvement is the better access to 3 types of accents and
   * 3 umlauts (for a, o and u).
   */
  public static final int DEMAP = 2;

  /**
   * Selects one keymap as currently active.
   * This value holds one of the keymap selection constants.
   * You can change it at runtime to select another than the
   * default US keymap.
   */
  public static int KMAP = USMAP;

  /* // *
   * KeyEvent objects need a source Component.
   * We use a dummy Container for this job.
   */
//  static final java.awt.Container compo = new java.awt.Container();

  /**
   * update / translate a char to reflect a new shift status
   */
  public static char xlate(char old, boolean shifted) {
    String numbers = "";
    if (KMAP == USMAP) numbers = ")!" +  "@#" + "$%^&*(";
    if (KMAP == UKMAP) numbers = ")!" + "\"" + "$%^&*(";
    if (KMAP == DEMAP) numbers = "=!" + "\"" + "$%&/()";
    if (Character.isISOControl(old)) return old;
      // (examples: shift tab, shift enter, shift escape,
      // shift delete?, shift backspace? control chars...)
    if (old == ' ') return old; // shift is senseless here
    if (Character.isDigit(old)) { // possibly shift a number
      if (!shifted) return old;
      try {
        return numbers.charAt(Character.digit(old, 10));
      } catch (Exception e) {
        // Wow, this language had more than 10 digits!
        // (instead of digit, getNumericValue is also a way)
        return old;
      }
    } // digits
    if (Character.isLetter(old)) {
      return (shifted ? Character.toUpperCase(old) :
        Character.toLowerCase(old));
    } // letters
    if (!shifted) { // remove shift from shift-number:
      int value = numbers.indexOf("" + old);
      if (value >= 0) return (char)('0' + value);
    } // un-shift numbers
    // now some (un)shifted versions of keys (at least for UK / US)
    if ((KMAP == USMAP) || (KMAP == UKMAP) || (KMAP == DEMAP)) {
      String lower = "-=[];\'`" + "\\" + ",./";
      String upper = "_+{}:\"~" +  "|" + "<>?";
      // UK uses QUOTE and BACKQUOTE keys for ~# and @'
      if (KMAP == UKMAP) {
        if ((old == '\'') || (old == '@'))
          return (shifted ? '@' : '\''); // US: ' "  UK: ' @
        if ((old == '#')  || (old == '~'))
          return (shifted ? '~' : '#');  // US: none UK: ~ #
        if ((old == '`')  || (old == ''))
          return (shifted ? '' : '`');  // US: ` ~  UK: ` 
        //   are unreachable on US. @ # are shift 2 3 on US.
      } // UK special stuff
      if (KMAP == DEMAP) { // German shifting
        // Umlauts are isLetter and are already handled, nice.
        upper = "?`" + "*" + "\'" + ">;:_";
        lower = "^ߴ" + "+" + "#"  + "<,.-";
        // ~ is on upper left corner, like US ` ~ and UK ` 
        // <> is right of left shift. Y and Z are swapped.
        // {[]}\\ @| are AltGr 7890 q< and AltGr 23m produce .
        // (UK / US do not even HAVE AltGr!). AltGr e is EURO.
        // At the place of -_ and =+ we have ? and ` (diacritics).
        // Shift-digits differs quite a lot. At the place of
        // [{ }] we have  +* and the \\| key is not there,
        // but instead we have a bigger return key.
        // Instead of ;: '" we have   and an additional
        // #' key similar to the UK ~ # key. The ,< .> keys
        // only differ partially and are ,; .: but the /?
        // key is a completely different -_ key here.
      } // DE special stuff
      if (!shifted) { // try to un-shift:
        int value = upper.indexOf("" + old);
        if (value >= 0) return lower.charAt(value);
      } // un-shift
      if (shifted) { // try to shift:
        int value = lower.indexOf("" + old);
        if (value >= 0) return upper.charAt(value);
      } // shift
      return old; // no further ideas on shifting!
    } // UK, US, DE
    else {
      System.out.println("Only UK, US and DE keyboards implemented!");
      return old;
    } // non UK and non US
  } // static char xlate(char old, boolean shifted)

  /**
   * update / translate a char given as keycode to reflect
   * a new shift status
   */
  public static char xlate(int old, boolean shifted) {
    char newch = ' ';
    // first, handle some standard cases:
    if ((old >= KeyEvent.VK_A) && (old <= KeyEvent.VK_Z))
      return xlate((char)old, shifted);
    if ((old >= KeyEvent.VK_0) && (old <= KeyEvent.VK_9))
      return xlate((char)old, shifted);
    if (old == KeyEvent.VK_SPACE) return ' '; // no shift here
    if (KMAP == UKMAP) { // special UK cases
      if (old == KeyEvent.VK_BACK_QUOTE)
        return (shifted ? '@' : '\'');
      if (old == KeyEvent.VK_QUOTE)
        return (shifted ? '~' : '#');
    }
    switch(old){
      case KeyEvent.VK_MINUS: { newch = '-'; break; }
        // MINUS and underscore: same key but other place for DE...
      case KeyEvent.VK_EQUALS: { newch = '='; break; }
      case KeyEvent.VK_OPEN_BRACKET: { newch = '['; break; }
      case KeyEvent.VK_CLOSE_BRACKET: { newch = ']'; break; }
      case KeyEvent.VK_SEMICOLON: { newch = ';'; break; }
      case KeyEvent.VK_QUOTE: { newch = '\''; break; }
      case KeyEvent.VK_BACK_QUOTE: { newch = '`'; break; }
      case KeyEvent.VK_PERIOD:
      case KeyEvent.VK_STOP: { newch = '.'; break; }
      case KeyEvent.VK_BACK_SLASH: { newch = '\\'; break; }
      case KeyEvent.VK_COMMA: { newch = ','; break; }
      case KeyEvent.VK_SLASH: { newch = '/'; break; }
      // cases which happen for neither UK nor US:
      // BACK_SPACE not needed
      case KeyEvent.VK_ESCAPE: { newch = 27; break; }
      // VK_TILDE not existing ???
      // Always shares with another VK, I hope. ` on US, # on UK.
      case KeyEvent.VK_EXCLAMATION_MARK: { newch = '!'; break; }
      case KeyEvent.VK_AT: { newch = '@'; break; }
      case KeyEvent.VK_NUMBER_SIGN: { newch = '#'; break; } // UK, DE
      case KeyEvent.VK_DOLLAR: { newch = '$'; break; }
      case KeyEvent.VK_DIVIDE: { newch = '%'; break; } // ?
        // or does divide mean the numpad key "/" ?
        // at least, VK_PERCENT does not exist anyway.
      case KeyEvent.VK_CIRCUMFLEX: { newch = '^'; break; } // for DE
      case KeyEvent.VK_AMPERSAND: { newch = '&'; break; }
      case KeyEvent.VK_ASTERISK: { newch = '*'; break; } // for DE!?
      case KeyEvent.VK_PLUS: { newch = '+'; break; }     // for DE!?
        // how about &szlig; and ? key in DE? Accent grave/... key?
      case KeyEvent.VK_LEFT_PARENTHESIS: { newch = '('; break; }
      case KeyEvent.VK_RIGHT_PARENTHESIS: { newch = ')'; break; }
      case KeyEvent.VK_BRACELEFT: { newch = '{'; break; }
      case KeyEvent.VK_BRACERIGHT: { newch = '}'; break; }
      case KeyEvent.VK_COLON: { newch = ':'; break; }
      case KeyEvent.VK_QUOTEDBL: { newch = '\"'; break; }
      // VK_PIPE ??? VK_QUESTION_MARK?
      case KeyEvent.VK_LESS: { newch = '<'; break; } // for DE!?
      case KeyEvent.VK_GREATER: { newch = '>'; break; } // for DE!?
      // how are DE keys for &szlig; and Umlauts called by Java???
      // GUK IM may need special AltGr treatment for {[]}\|@~ ...
    }
    if (newch != ' ') return xlate(newch, shifted);

    newch = '?';
    // could do further processing here
    return newch;
  } // static char xlate(int old, boolean shifted)

 /**
  * for testing
  */ 
 public static void main(String[] args) {
   char which;
   try {
     KMAP = Integer.parseInt(args[0].trim());
   } catch (Exception e) {
     System.out.println("Give " + UKMAP + ", " + USMAP + " or "
       + DEMAP + " as argument to test UK, US or DE shift"
       + "mapping.\nDefault is " + KMAP + ".");
   }
   for (which = ' '; which <= (char)127; which++) {
     System.out.print("" + which + "["
       + xlate(which, false) + "" + xlate(which, true) + "]   ");
   }
   System.out.println("\n**************");
   for (which = ' '+(char)128; which <= (char)255; which++) {
     System.out.print("" + which + "["
       + xlate(which, false) + "" + xlate(which, true) + "]   ");
   }
   System.out.println();
 } // void main(String[] args)

} // class ShiftedKeys

