#!/usr/bin/perl

# Public Domain filter by Eric Auer 6/2005: Send histogram to gnuplot.

# Input file example:
# abcdefgh.text.1np1      -1 39.8 80.4 61.9 27.1 11.6  3.7  2.4  2.7  2.7
# abcdefgh.text.2verb     -1 10.1  4.7 25.4 41.1 28.4 18.7 11.7  5.5  5.1 
# abcdefgh.text.3adv      -1  2.4  0.9 5.8  21.5 32.3 31.0 24.7 17.0 10.4

# Needs gnuplot. Suggested log filter calling style:
# perl histplot.pl < pooled-abcdefgh.log
# creates [region].tmp files and some [name of abcdefgh].eps file!
# tmp files are deleted after gnuplot processing is done.

# Thanks to http://www.cs.uni.edu/Help/gnuplot/ ...
# and http://www.gnuplot.info/docs/gnuplot.html

my $cutoff = 25;	# at most first 25 fixations
			# ... plot becomes too sparse / noisy further right
my $ylimit = 70;	# clip graph at ? percent: 50, 70, 80, 88 or 100...
my $usecolor = 1;	# color versus monochrome eps files
my $usesort = 1;	# use fixed order of sub-plots

my %groupfilenames = ("abcdefgh", "1-all",
  "abcd", "2a-matching",
  "efgh", "2a-nonmatching",
  "abgh", "2b-to-left-action",
  "cdef", "2b-to-right-action",
  "aceg", "2c-SVO",
  "bdfh", "2c-OVS",
  "ab", "4a-match-to-left",
  "cd", "4a-match-to-right",
  "gh", "4a-nomatch-to-left",
  "ef", "4a-nomatch-to-right",
  "ag", "4b-SVO-to-left",
  "ce", "4b-SVO-to-right",
  "bh", "4b-OVS-to-left",
  "df", "4b-OVS-to-right",
  "ac", "4c-match-SVO",
  "eg", "4c-nomatch-SVO",
  "bd", "4c-match-OVS",
  "fh", "4c-nomatch-OVS");

my %groupnames = ("abcdefgh", "All trialtypes",
  "abcd", "Trials with matching image and text",
  "efgh", "Trials with nonmatching image and text",
  "abgh", "Trials with images with to the left action",
  "cdef", "Trials with images with to the right action",
  "aceg", "Trials with normal SVO word order",
  "bdfh", "Trials with OVS word order",
  "ab", "Match / to the left trials",
  "cd", "Match / to the right trials",
  "gh", "Nomatch / to the left trials",
  "ef", "Nomatch / to the right trials",
  "ag", "normal SVO / to the left trials",
  "ce", "normal SVO / to the right trials",
  "bh", "OVS / to the left trials",
  "df", "OVS / to the right trials",
  "ac", "normal SVO / match trials",
  "eg", "normal SVO / nomatch trials",
  "bd", "OVS / match trials",
  "fh", "OVS / nomatch trials");

my %linestyles = (
  "text.any", "linetype 7 pointtype 2 pointsize 1.4",
  # "black.any", "linetype 7 pointtype 1",
  "black.any", "linetype 7",
  "image.any", "linetype 7 pointtype 3 pointsize 1.4",
  "text.1np1", "linetype 1 pointtype 1 pointsize 1",
  "text.2verb", "linetype 1 pointtype 6 pointsize 0.8",
  "text.3adv", "linetype 1 pointtype 9 pointsize 1.4",
  "text.4np2", "linetype 1 pointtype 11 pointsize 1.4",
  "image.agt", "linetype 2 linewidth 2 pointtype 3 pointsize 1.2",
  "image.2", "linetype 2 linewidth 2 pointtype 1 pointsize 1.5",
  "image.2b", "linetype 2 linewidth 2 pointtype 1 pointsize 1.5",
  "image.pat", "linetype 2 linewidth 2 pointtype 7 pointsize 1.2",
  "image.1", "linetype 3 pointtype 8 pointsize 0.6",
  "image.3", "linetype 3 pointtype 10 pointsize 0.6",
  "image.tx", "linetype 3",
  "image.ta", "linetype 3 pointtype 6 pointsize 0.8",
  "image.tp", "linetype 3 pointtype 4 pointsize 0.8"
  );
  # linetype sets the color and dashing style
  # lines:  0 dashed ("grid color") 1 red 2 green 3 blue 7 gray (?)...
  #   x11: 0 dashed-black -1 black 4 magenta 5 cyan 6 brown 7 orange...
  # points: 2 cross 1 plus 3 star
  #         4 box 5 filledbox 6 circle 7 filledcircle
  #         8 up 9 filledup 10 down 11 filleddown

my %altstyles = (
  "image.ta", "linetype 3 pointtype 8 pointsize 0.6",
  "image.tp", "linetype 3 pointtype 10 pointsize 0.6",
  "image.tx", "linetype 3 pointtype 4 pointsize 0.6"
  ); # used if image.1 / image.3 are not shown: less in-the-way
  # style to show text-semantically labelled image things...

my @items = (
  "black.any",
  "text.any",
  "image.any",
  "image.agt", "image.2", "image.pat",
  "text.1np1", "text.2verb", "text.3adv", "text.4np2",
  "image.1", "image.3",
  "image.tx", "image.ta", "image.tp"
  );
# black reaches 20-35 percent on 2nd fixation, 9-17 percent on 3rd
# fixation, 3-7 percent on 4th fixation, not very interesting in general.
# select EITHER image.2 OR image.2b - put only ONE of them in @items...

my %tmpfiles = ();

# default dashes are "all solid" (except grid) for color display...
# default -gray are 100 60  80 40  90 50  70 30
# default line colors are: red green blue magenta cyan sienna orange coral

# This does not work: -line2Color green,0.5 -line5Color cyan,0.5
#                     -line6Color yellow,0.5 -line7Color DarkOrange1"
# ... so you need either the hack at the bottom or an xresources file.

open (PLOT,"|gnuplot") || die "Gnuplot required\n";

# lines or boxes, steps, dotslines, points, impulses
print PLOT "set grid\nset data style lines\nset yrange [ 0 : $ylimit ]\n";
print PLOT "set missing \"?\"\nset xrange [ 1 : " . $cutoff . " ]\n";
print PLOT "set xlabel \"fixation number\"\nset ylabel \"percentage\"\n";
# either use default key "top right inside with xrange to cutoff+5
# or use "set key outside" or "below" to move the key outside the graph
print PLOT "set key outside spacing 3\n";
# print PLOT "set key below\n";
print PLOT "set xtics 1\n";	# write all fixation numbers
print PLOT "set ytics 10\nset mytics 2\n";
# you could also use "set xtics (1,5,10,15,20,25)"...
# set mxtics N selects N-1 minor ticks between each pair of major ticks


my $type = "";
my %data = ();
my $which = 0;
foreach (<STDIN>) {
  my $line = $_;
  chomp($line);
  $line =~ tr/\t/ /;
  my @base = ();
  my ($key, @values) = split(/[ ]+/,$line);
  ($type, @base) = split(/[.]/, $key);
  my $basekey = join('.',@base);
  $data{$basekey} = join(' ',@values);
  if ($usesort == 0) {	# preserve item order from file?
    $items[$which] = $basekey;
    $which++;
  }
}

if (defined($groupnames{$type})) {
  print PLOT "set title \"" . $groupnames{$type} . " (types $type)\"\n";
} else {
  print PLOT "set title \"Trial types $type\"\n";
}

my $epsfilename = "histo-x-" . $type . ".eps";
if (defined($groupfilenames{$type})) {
  $epsfilename = "histo-" . $groupfilenames{$type} . ".eps";
}

$which = 0;
print PLOT "plot ";
foreach $basekey (@items) {
  if (!defined($data{$basekey})) {
    die "Oops, no data for key $basekey\n";
  }
  my @values = split(/[ ]/,$data{$basekey});

  my $how = "linespoints";
  if ($basekey =~ /^text.[1-4]/) {
    # "lines", "boxes", "steps", "linespoints", "points", "impulses"
    $how = "linespoints";	# specific text areas
  }
  if ($basekey =~ /^image.[13]/) {
    $how = "points";		# image location labelled images
  }
  if ($basekey =~ /any$/) {
    $how = "points";		# pool categories
  }
  if ($basekey =~ /^image.t/) {
    $how = "points";		# text-semantically labelled images
    if ($basekey eq "image.tx") {
      $how = "impulses";
    }
  }
  if ($basekey eq "black.any") {
    $how = "boxes";		# usually quite low: background stuff
  }

  if ( defined($altstyles{$basekey}) &&
    ($basekey =~ /2b-/) || ($basekey =~ /4[ab]-/) ) {
    # use alternative (less visible) style for image.t*
    # in cases where image.[13] are not in use...
    $how = "points " . $altstyles{$basekey};
  } else {
    if (defined($linestyles{$basekey})) {
      $how .= " " . $linestyles{$basekey};
    }
  }

  if (
    ( (($epsfilename =~ /left/) || ($epsfilename =~ /right/)) &&
      ($basekey =~ /^image.[13]/) ) ||
    ( ($epsfilename =~ /SVO/) && ($basekey eq "image.ta") ) ||
    ( ($epsfilename =~ /OVS/) && ($basekey eq "image.tp") ) ||
    ( ($epsfilename =~ /4c-/) && ($basekey =~ /image.t/) ) ) {
    # image.1 / image.3 is just the same as image.agt or image.pat
    # (in one of both possible ways) if we plot for fixed direction
    # image.ta is just the same as image.2(b) for SVO and
    # image.tp is just the same as image.2(b) for OVS...
    # image.t? are fixed to agent/patient/2(b) if SVO/OVS and
    # match/nomatch are both fixed for the whole plot...
  } else {
    if ($which > 0) {
      print PLOT " \\\n  ,";
    }
    if ($basekey eq "black.any") {
      # print PLOT "\"$basekey.tmp\" notitle with $how";
      print PLOT "\"$basekey.tmp\" title \"$basekey\" with $how";
    } else {
      # smooth csplines / sbezier / bezier: bezier modifies curves too much
      #   and csplines causes too many point marks in linespoints mode...
      # print PLOT "\"$basekey.tmp\" smooth cspline title \"$basekey\" with $how";
      print PLOT "\"$basekey.tmp\" title \"$basekey\" with $how";
    }
    open(TMPFILE,">$basekey.tmp") || die "cannot write $basekey.tmp\n";
    $tmpfilenames{"$basekey.tmp"} = 1;
    for $n (1 .. $#values) {
      if ($n <= $cutoff) {
        if ($values[$n-1] > -1) {
          print TMPFILE "$n " . $values[$n-1] . "\n";
        } else {
          print TMPFILE "$n ?\n";	# as defined by set missing
        }
      }
    } # create tempfile
    close TMPFILE;
  } # item which we want to show
  $which++;
} # loop over items

print PLOT "\nset output \"$epsfilename\"\n";
# next line is only useful if using "gnuplot gnuplot.script"
# print PLOT "pause -1 \"Hit return to save to postscript\"\n";

if ($usecolor > 0) {
  print PLOT "set terminal postscript eps color solid\n";
  # print PLOT "set terminal png color\n";
  # png font: default small, or use medium... colors: color, gray, monochrome
} else {
  print PLOT "set terminal postscript eps monochrome solid\n";
  # print PLOT "set terminal png monochrome\n";
} # add "solid" option for "postscript" to avoid dashed lines

print PLOT "replot\nexit\n";
close PLOT;

foreach $x (keys %tmpfilenames) {
  unlink $x;	# delete tempfiles again
}

# we either have to set the lineNColor values in our xresources - or cheat:
if (open (PSFILE,"<$epsfilename")) {
  my @psdata = (<PSFILE>);
  close(PSFILE);
  open (PSFILE,">$epsfilename") || die "Oops, cannot write modified EPS\n";
  foreach $line (@psdata) {
    chomp($line);
    if ($line =~ /LT.*def/) {
      $line =~ s/0[.]3/0.5/;		# fix orange green-ness
      $line =~ s/0 1 0/0 0.5 0/;	# make green less bright
      $line =~ s/0 1 1/0 0.5 0.5/;	# make cyan less bright
      $line =~ s/1 1 0/0.5 0.5 0/;	# make yellow less bright
    }
    print PSFILE "$line\n";
  }
}
