import java.awt.image.*;
import java.awt.*;
import java.io.*;

public class BMPLoader 
{
    
    // build an int from a byte array - convert little to big endian
    public static int constructInt(byte[] in,int offset) {
        int ret =          ((int)in[offset + 3] & 0xff);
        ret = (ret << 8) | ((int)in[offset + 2] & 0xff);
        ret = (ret << 8) | ((int)in[offset + 1] & 0xff);
        ret = (ret << 8) | ((int)in[offset + 0] & 0xff);
        return(ret);
    }
    // build an int from a byte array - convert little to big endian
    // set high order bytes to 0xfff
    public static int constructInt3(byte[] in,int offset) {
        int ret =            0xff;
        ret = (ret << 8) | ((int)in[offset + 2] & 0xff);
        ret = (ret << 8) | ((int)in[offset + 1] & 0xff);
        ret = (ret << 8) | ((int)in[offset + 0] & 0xff);
        return(ret);
    }
   
    // build an short from a byte array - convert little to big endian
    public static short constructShort(byte[] in,int offset) {
        short ret =        (short)((short)in[offset + 1] & 0xff);
        ret = (short)((ret << 8) | (short)((short)in[offset + 0] & 0xff));
        return(ret);
    }
    
    //
    // internal class representing a bitmap header structure 
    // with code to read it from a file
    static class BitmapHeader {
	public int nmagic;
        public int nsize;
        public int nbisize;
        public int nwidth;
        public int nheight;
        public int nplanes;
        public int nbitcount;
        public int ncompression;
        public int nsizeimage;
        public int nxpm;
        public int nypm;
        public int nclrused;
        public int nclrimp;
        
        // read in the bitmap header
        public void read(FileInputStream fs) throws IOException 
        {
            final int bflen=14;  // 14 byte BITMAPFILEHEADER
            byte bf[]=new byte[bflen];
            fs.read(bf,0,bflen);
            final int bilen=40; // 40-byte BITMAPINFOHEADER
            byte bi[]=new byte[bilen];
            fs.read(bi,0,bilen);

            // Interpret data.
	    nmagic = constructInt(bf,0);	// should be B, M bytes
            nsize = constructInt(bf,2);
           //  System.out.println("File type is :"+(char)bf[0]+(char)bf[1]);
          //  System.out.println("Size of file is :"+nsize);

            nbisize = constructInt(bi,0);	// ,2 was wrong
         //   System.out.println("Size of bitmapinfoheader is :"+nbisize);

            nwidth = constructInt(bi,4);
         //   System.out.println("Width is :"+nwidth);

            nheight = constructInt(bi,8);
          //  System.out.println("Height is :"+nheight);

             nplanes = constructShort(bi,12); //(((int)bi[13]&0xff)<<8) | (int)bi[12]&0xff;
          //  System.out.println("Planes is :"+nplanes);

             nbitcount = constructShort(bi,14); //(((int)bi[15]&0xff)<<8) | (int)bi[14]&0xff;
          //  System.out.println("BitCount is :"+nbitcount);

            // Look for non-zero values to indicate compression, e.g. 1 for RLE
             ncompression = constructInt(bi,16); 
          //  System.out.println("Compression is :"+ncompression);

             nsizeimage = constructInt(bi,20); 
          //  System.out.println("SizeImage is :"+nsizeimage);

             nxpm = constructInt(bi,24); 
           // System.out.println("X-Pixels per meter is :"+nxpm);

             nypm = constructInt(bi,28); 
           // System.out.println("Y-Pixels per meter is :"+nypm);

             nclrused = constructInt(bi,32); 
          //  System.out.println("Colors used are :"+nclrused);

             nclrimp = constructInt(bi,36); 
          //  System.out.println("Colors important are :"+nclrimp);
        }
    }
    
  public static Image read(FileInputStream fs) 
  {
	Image img = null;
         try {
            BitmapHeader bh = new BitmapHeader();
            bh.read(fs);
 
	    if ( (bh.ncompression != 0) || (bh.nplanes != 1) ||
		 (bh.nmagic != ( ((int)'M' << 8) | (int)'B' ) ) ) {
		System.err.println("Sorry, only uncompressed 1 plane BMP supported!");
	    } else {
                if (bh.nbitcount==24)
                    img = readMap24(fs,bh);
                if (bh.nbitcount==8)
                    img = readMap8(fs,bh);
	    }
            fs.close();
	    if (img == null)
		System.err.println("Unsupported BMP file type!");
	    return img;	// unsupported type!
        }
        catch (IOException e)
        {
          System.out.println("File access error in BMPLoader.");
        }
        return(null);
    } 

    /**
        readMap24 internal routine to read the bytes in a 24 bit bitmap
        
        Arguments:
           fs - file stream
           bh - header struct
        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */

  protected static Image readMap24(FileInputStream fs,BitmapHeader bh) throws IOException
  {
        Image image;
          // No Palatte data for 24-bit format but scan lines are
        // padded out to even 4-byte boundaries.
        int npad = (bh.nsizeimage / bh.nheight) - bh.nwidth * 3;
        int ndata[] = new int [bh.nheight * bh.nwidth];
        byte brgb[] = new byte [( bh.nwidth + npad) * 3 * bh.nheight];
        fs.read (brgb, 0, (bh.nwidth + npad) * 3 * bh.nheight);
        int nindex = 0;
        for (int j = 0; j < bh.nheight; j++)
            {
            for (int i = 0; i < bh.nwidth; i++)
                {
                ndata [bh.nwidth * (bh.nheight - j - 1) + i] = constructInt3(brgb,nindex);
                 //   (255&0xff)<<24
                 //   | (((int)brgb[nindex+2]&0xff)<<16)
                 //   | (((int)brgb[nindex+1]&0xff)<<8)
                 //   | (int)brgb[nindex]&0xff;
                // System.out.println("Encoded Color at ("
               //     +i+","+j+")is:"+nrgb+" (R,G,B)= ("
                //    +((int)(brgb[2]) & 0xff)+","
                //    +((int)brgb[1]&0xff)+","
                //    +((int)brgb[0]&0xff)+")");
                nindex += 3;
                }
            nindex += npad;
            }

        image = Toolkit.getDefaultToolkit().createImage
            ( new MemoryImageSource (bh.nwidth, bh.nheight,
                                     ndata, 0, bh.nwidth));
        return(image);
    }

    /**
        readMap8 internal routine to read the bytes in a 8 bit bitmap
        
        Arguments:
           fs - file stream
           bh - header struct
        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */
    protected static Image readMap8(FileInputStream fs,BitmapHeader bh) throws IOException
      {
        Image image;

                // Have to determine the number of colors, the clrsused
                // parameter is dominant if it is greater than zero.  If
                // zero, calculate colors based on bitsperpixel.
                int nNumColors = 0;
                if (bh.nclrused > 0)
                    {
                    nNumColors = bh.nclrused;
                    }
                else
                    {
                    nNumColors = (1&0xff)<< bh.nbitcount;
                    }
           //     System.out.println("The number of Colors is"+nNumColors);

                // Some bitmaps do not have the sizeimage field calculated
                // Ferret out these cases and fix 'em.
                if (bh.nsizeimage == 0)
                    {
                    bh.nsizeimage = ((((bh.nwidth* bh.nbitcount)+31) & ~31 ) >> 3);
                    bh.nsizeimage *= bh.nheight;
          //          System.out.println("nsizeimage (backup) is"+nsizeimage);
                    }

                // Read the palatte colors.
                int  npalette[] = new int [nNumColors];
                byte bpalette[] = new byte [nNumColors*4];
                fs.read (bpalette, 0, nNumColors*4);
                int nindex8 = 0;
                for (int n = 0; n < nNumColors; n++)
                    {
                    npalette[n] = constructInt3(bpalette,nindex8);
                    nindex8 += 4;
                    }

                // Read the image data (actually indices into the palette)
                // Scan lines are still padded out to even 4-byte
                // boundaries.
                int npad8 = (bh.nsizeimage / bh.nheight) - bh.nwidth;
            //    System.out.println("nPad is:"+npad8);

                int  ndata8[] = new int [bh.nwidth * bh.nheight];
                byte bdata[] = new byte [(bh.nwidth+npad8)* bh.nheight];
                fs.read (bdata, 0, (bh.nwidth+npad8)*bh.nheight);
                nindex8 = 0;
                for (int j8 = 0; j8 < bh.nheight; j8++)
                    {
                    for (int i8 = 0; i8 < bh.nwidth; i8++)
                        {
                        ndata8 [bh.nwidth*(bh.nheight-j8-1)+i8] =
                            npalette [((int)bdata[nindex8]&0xff)];
                        nindex8++;
                        }
                    nindex8 += npad8;
                    }

                image = Toolkit.getDefaultToolkit().createImage
                    ( new MemoryImageSource (bh.nwidth, bh.nheight,
                                             ndata8, 0, bh.nwidth));

            return(image);
         }

    /**
        load method - see read for details
        
        Arguments:
            sdir and sfile are the result of the FileDialog()
            getDirectory() and getFile() methods.

        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */
    public static Image  load(String sdir, String sfile) {
		return(load(sdir + sfile));
	}

 
    /**
        load method - see read for details
        
        Arguments:
            sdir - full path name

        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */
    public static Image  load(String sdir) 
    {
	FileInputStream fs;
        try
        {
            fs = new FileInputStream(sdir);
	}
	catch (IOException ex) {
	    System.out.println("Could not open BMP file <" + sdir + ">");
	    return(null);
	}
        return(read(fs));
    }

// end class BMPLoader
}
