import java.awt.image.*;
import java.awt.*;
import java.io.*;	// RandomAccessFile File, various Exceptions

public class PCXLoader /* by EA 5/2005 - based on BMPLoader */
{

    /** build a palette int from a PCX R, G, B palette entry in a  */
    /* byte array. high ... low bytes in each int are: 0xff R G B. */
    public static int constructPal(byte[] in, int offset) {
        int ret = 0xff;					// ff
	ret = (ret << 8) | (in[offset + 0] & 0xff);	//    R
        ret = (ret << 8) | (in[offset + 1] & 0xff);	//      G
        ret = (ret << 8) | (in[offset + 2] & 0xff);	//        B
        return(ret);
    }

    /** build a short int from a little endian value in a byte array */
    public static int constructShort(byte[] in, int offset) {
        int ret =          (in[offset + 1] & 0xff);	// 0xff: no sign please
        ret = (ret << 8) | (in[offset + 0] & 0xff);
        return(ret);
    }

    //
    // internal class representing a PCX header structure
    // with code to read it from a file
    static class PCXHeader {
        public int nwidth;
        public int nheight;
        public int nplanes;	/* only 1 supported by readPcx8 yet */
        public int nbitcount;	/* only 8 supported by readPcx8 yet */

        // read in the 128 byte PCX header
        public void read(byte [] contents)
        {
	    nwidth = 1 + constructShort(contents, 8) -
		constructShort(contents, 4); // "right - left end"
	    nheight = 1 + constructShort(contents, 10) -
		constructShort(contents, 6); // "top - bottom end"
	    nplanes = contents[65];	// 1 for normal 4-8 bit formats
	    nbitcount = contents[3];
	    if ((contents[2] != 1) ||	// 1 for normal RLE compression
		(contents[0] != 10) ||	// 10 for normal PCX format
		(contents[1] < 5))	// 5 or newer required
		nbitcount = 0;		// flag trouble ahead

	    // ignored: X/Y dots per inch info words at offset 12-15
	    // 16-of-64 color EGA palette at offset 16-63, palette
	    // type word at offset 68-69 (deprecated), bytes-per-line word
	    // (as width) at offset 66-67. Offset 64 and 70-127 are unused.
        }
    }

  public static Image read(RandomAccessFile raf)
  {
	byte [] contents;
        int [] palette = new int[256];
	int i, size;

        try {
	    size = (int)(raf.length());		// can be long (64 bits)
            contents = new byte[size];		// read file to RAM
	    raf.readFully(contents);
	    raf.close();
        }
        catch (IOException e)
        {
	    System.err.println("Caught exception in PCX loader:\n" + e.toString());
	    return(null);
        }
	if (size < (128+768+1))		/* header + palette */
	    return(null);		/* too small file */

        PCXHeader ph = new PCXHeader();
        ph.read(contents);

        if ((ph.nbitcount!=8) ||	/* 8 bit per pixel file? */
	    (ph.nplanes!=1)) {		/* 1 plane? */
	    System.err.println("Only 256 color PCX supported (8 bit/pixel, 1 plane)");
            return null;		/* others not supported yet */
	}

	i = size - (768 + 1);		/* start of palette - 1 */
	if (contents[i] != 12) {	/* no palette magic found? */
	   for (int j = 0; j < 256; j++)
		palette[j] = j | (j << 8) | (j << 16) | 0xff000000;
		/* construct a greyscale palette from scratch */
	} else {			/* read palette */
	    i++;
	    for (int j = 0; j < 256; j++) {
		palette[j] = constructPal(contents, i);
		i += 3;
	    }
	}

	return(readPcx8(contents, ph, palette));
    }

    /**
        readPcx8 internal routine to read the bytes in a 8 bit PCX image

        Arguments:
            contents - byte array, contents of whole file
            ph - header object
	    palette - int array, 256 entries

        Returns:
            Image Object, be sure to check for (Image)null

    */
    protected static Image readPcx8(byte [] contents, PCXHeader ph, int [] palette)
    {
        Image image;
	int [] canvas;
	int i, x, y;

	if ((contents == null) || (ph == null) || (palette == null))
	    return null;
	if ((contents.length < 128 + ph.nheight) || (palette.length < 256))
	    return null;	/* unuseable file / palette size */

	canvas = new int [ph.nwidth * ph.nheight];

	i = 128;	// location of data stream in file
	x = 0;
	y = 0;

        do {
	    int color = contents[i] & 0xff;	// color or RLE start byte
	    i++;
	    if ((color & 0xc0) == 0xc0) {	// start of RLE run?
		int count = color & 0x3f;
		color = contents[i] & 0xff;	// actual color (256 colors)
		i++;
		for ( ; count>0; count--) {	// write up to 63 pixels
		    canvas[(ph.nwidth*y)+x] = palette[color];
		    x++;			// RLE runs never wrap lines
		}
	    } else {				// single pixel (192 colors)
		canvas[(ph.nwidth*y)+x] = palette[color];
		x++;
	    }
	    if (x >= ph.nwidth) {		// line full?
		x = 0;
		y++;
	    }
        } while (y < ph.nheight);

	// MemoryImageSource input is an array of int:
	// high ... low bytes in each int are: 0xff R G B 
        image = Toolkit.getDefaultToolkit().createImage
            ( new MemoryImageSource (ph.nwidth, ph.nheight,
                  canvas, 0, ph.nwidth));

        return(image);
    }

    /**
        load method - see read for details

        Arguments:
            sdir and sfile are the result of the FileDialog()
            getDirectory() and getFile() methods.

        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */
    public static Image  load(String sdir, String sfile) {
		return(load(sdir + sfile));
	}

    /**
        load method - see read for details

        Arguments:
            sdir - full path name

        Returns:
            Image Object, be sure to check for (Image)null !!!!

    */
    public static Image  load(String sdir)
    {
        try
        {
	    Image img;
            RandomAccessFile raf=new RandomAccessFile(new File(sdir), "r");
            img = read(raf);
	    if (img == null)
		System.err.println("Unsupported PCX image type: " + sdir);
	    return img;
	} catch (FileNotFoundException fnfe) {
	    System.err.println("PCX image file not found: " + sdir);
	    return(null);
	} catch(IOException ioe) {
	    System.err.println("I/O Error loading PCX image: " + sdir);
	    return(null);
	}
    }

} // end of PCXLoader class

