/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* Detect fixation and saccade start. Log start / end times of   */
/* fixations, and (start) location of fixations. Uses 1 kHz FIFO */
/* Saccade: at least 4 times in a row min. movement 1? pixel.    */
/* Fixation: at least 5 times in a row max. movement 2? pixels.  */
/* fifo[0] vs. fifo[1] for saccade, and [0] vs [5] for fixation. */
/* Do not log fixation if below ?? msec. Units assume 8x16 font. */
/* UPDATE 4/2005: Allow font zooming for VESA mode, with FONT_ZOOM */
/* UPDATE 4/2005: Raised min logged duration from 1 to 5 msec     */
/* UPDATE 4/2005: Modified fixation time log @: back-date start, */
/* fix duration calculation rounding. Obsoletes postprocessing. */


#include "tracker.h"	/* global headers */
#include "script.h"	/* LOG_STRING and similar */
#include <math.h>	/* fabs, powi, sqrt... */

#if GRAPHICS
#include "vesa.h"	/* for lfbSel for FONT_ZOOM */
#endif

#define square(x) powi((x),2)	/* fast "x * x" */

/* #define Manhattan(lag) ( abs((int)(fifo_x[0]-fifo_x[lag])) + \ */
/*   abs((int)(fifo_y[0]-fifo_y[lag])) )                          */

#define DistSquared(lag) ( square(0.125 * (fifo_x[0]-fifo_x[lag])) + \
  square(0.0625 * (fifo_y[0]-fifo_y[lag])) )
  /* square of the real distance in characters */

#define FIFO_LEN 10	/* all windows have to be at most FIFO_LEN long */


#define SACC_WINDOW_MOUSE 10	/* use as longer timespan for saccade check */
			/* needed because mice have < 1000 samples / sec */
#define SACC_WINDOW 1	/* (1) measurement window size, in msec */
#define SACC_MINMOVE (1.0/6.0)	/* (1/6?) speed in chars / msec */
#define SACC_MINTIME 4	/* (3) samples during which eye has to keep moving  */

#define FIX_WINDOW 5	/* (5) measurement window size, in msec */
#define FIX_MAXMOVE (2.0/6.0)	/* (2/6) speed in chars / msec */
#define FIX_MINTIME 5	/* (5) samples during which gaze has to stay stable */
#define FIXLOG_MINTIME 5	/* (60) do not log fixations below ... msec */
	/* UPDATE 4/2005: FIXLOG_MINTIME changed from 1 to 5, will throw  */
	/* out avg. 10 msec of fixations in a 5 second trial, fair enough */
	/* Fix. of 5 to 59 msec are post-processor "pooling" candidates.  */


/* check coordinate progress and log fixations when detected */
/* pass a zero firsttime value to start initialization process */
void check_fixations(uint32 firsttime)	/* firsttime is a timestamp */
{
  static double fifo_x[FIFO_LEN+1];
  static double fifo_y[FIFO_LEN+1];
  static uint32 tstamp;	/* limits FIFO fill frequency */

  static int stable;	/* count of consecutive "gaze" detections */
  static int moving;	/* count of consecutive "move" detections */

  static uint32 fixation_start;		/* in msec */
  static int fixation_x;
  static int fixation_y;

  uint32 timestamp, now_msec;	/* timestamp, in ysec and in msec */

  timestamp = get_ytime ();		/* in ysec */

  if (!firsttime) {			/* initialize */
    stable = -(FIFO_LEN+1);		/* FIFO_LEN steps to go until start */
    moving = 0;
    tstamp = timestamp;			/* wait for first sample */
    fixation_x = -1;			/* wait for first fixation */
    fixation_start = timestamp / 1000U;
  }

  if ((timestamp - tstamp) < 1000U)
    return;				/* not yet 1 msec since last sample */
  tstamp = timestamp;			/* block updates for 1 msec */

  now_msec = timestamp / 1000U;

  {
    int i = FIFO_LEN;

    while (i > 0) {
      fifo_x[i] = fifo_x[i-1];
      fifo_y[i] = fifo_y[i-1];
      i--;
    }
  }

  { /* fetch exact gaze coordinates */
    double f_x, f_y;

    get_pixel_xy (&f_x, &f_y);
    fifo_x[0] = f_x;
    fifo_y[0] = f_y;
  } /* this is fast, it just reads coordinates from buffer updated by get_xy() */

  if (stable < 0) {			/* still in FIFO fill process? */
    stable++;
    return;
  }

  if ( ( using_mouse() && ( DistSquared(SACC_WINDOW_MOUSE) >=
      square(SACC_WINDOW_MOUSE * SACC_MINMOVE) ) ) ||
       ( !using_mouse() && ( DistSquared(SACC_WINDOW) >=
      square(SACC_WINDOW * SACC_MINMOVE) ) ) ) {

    moving++;				/* printf("MMM"); */
    if (moving >= SACC_MINTIME) {
      if (fixation_x != -1) {		/* saccade ends a known fixation */
        if ( (now_msec - fixation_start) < (SACC_MINTIME + FIXLOG_MINTIME) ) {
          /* LOG_STRING("*short*"); */	/* too short to be logged */
        } else {
          char fixstring[80];
          uint32 first_msec, now_rel_msec;			/* @ */

          first_msec = fixation_start - (firsttime/1000U);	/* make rel. */
          timestamp -= (1000U * SACC_MINTIME);	/* @ cheat: back-date NOWtime */
          /* because saccade start is detected SACC_MINTIME after it happens! */
          now_rel_msec = (timestamp - firsttime)/1000U;		/* @ */

          /* Values are X, Y, starttime, duration (in pixels and msecs) */
          /* timestamp of string itself is calculated like now_rel_msec */
          sprintf(fixstring,"FIXATION %d %d %u %u", fixation_x, fixation_y,
            first_msec, now_rel_msec - first_msec);		/* @ */
          /* Using (now_msec - fixation_start) could cause rounding errors */
          LOG_STRING(fixstring);	/* uses timestamp, firsttime and */
					/* logleft, logptr from script.h */

	  timestamp += (1000U * SACC_MINTIME);	/* @ un-do back-date cheat */
	}
        fixation_x = -1;		/* data used, wait for next */
      } else {
        /* LOG_STRING("*nodata*"); */	/* saccade ends no fixation */
      }
    
    } /* saccade starting, possible end of a fixation */

  } else {

    moving = 0;				/* printf("---"); */

  }

  if (DistSquared(FIX_WINDOW) <= square(FIX_WINDOW * FIX_MAXMOVE)) {
    stable++;				/* printf("SSS\r"); */
    if ( (stable >= FIX_MINTIME) && (fixation_x == -1) ) {	/* start! */

#if GRAPHICS				/* new 4/2005: FONT_ZOOM */
      double fzoom = (lfbSel > 0) ? FONT_ZOOM : 1.0;	/* only if VESA */
      fixation_x = (int)((fifo_x[0] * fzoom) + 0.5);
      fixation_y = (int)((fifo_y[0] * fzoom) + 0.5);
#else
      fixation_x = (int)(fifo_x[0] + 0.5);
      fixation_y = (int)(fifo_y[0] + 0.5);
#endif
      fixation_start = now_msec - FIX_MINTIME;	/* "middle" time of settle */
      /* looking back: endtime of FIX_MINTIMEth sample window, the samples */
      /* from (this time - FIX_WINDOW msec) on all are fixation evidence.  */

    } /* fixation */
  } else {
    stable = 0;				/* printf("---\r"); */
  }
  
} /* check_fixations */

